<?php
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Resources\AuthorisedMobileNumber as MobileNumber;
	
	require("vendor/autoload.php");
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Environment
	*/
	
	$whitelist = array(
		"127.0.0.1",
		"::1"
	);

	if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
		define("ENVIRONMENT", "development");
	} else {
		define("ENVIRONMENT", "production");
	}
	
	/*
		Middleware
	*/
	
	$app->add(function ($req, $res, $next) {
	    $response = $next($req, $res);
	    return $response
			->withHeader('Access-Control-Allow-Origin', '*')
			->withHeader('Access-Control-Allow-Headers', 'X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id, Vinteract-Access-Token')
			->withHeader('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, PATCH, OPTIONS');
	});
	
	$app->add(function($request, $response, $next) {
		
		// Init client?
		
		$headers = $request->getHeader("Vinteract-Client-Id");
		
		if (count($headers)) {
			Vinteract\Client::init($headers[0]);
		}
		
		if (ENVIRONMENT === "development") {
			Vinteract\Client::init("EjNic");
		}
		
		// Return response.
		
		return $next($request, $response);
		
	});
	
	/*
		Send SMS
		TODO Validate mobile number format
		TODO Throttle requests
	*/
	
	$app->post("/", function(Request $request, Response $response, array $args) {
		
		$data = $request->getParsedBody();
		
		if (isset($data["mobile_number"]) and isset($data["message"])) {
			
			$sms = new Vinteract\Notifications\SMS();
			
			$smsResponse = $sms->send([
				"mobile_number" => $data["mobile_number"],
				"message" => $data["message"],
			]);
			
			return $response->withJson($smsResponse->getResponseObject());
			
		}
		
		return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		
	});
	
	/*
		Authorised Mobile Numbers
	*/
	
	$app->get("/authorised-mobile-numbers", function(Request $request, Response $response, array $args) {
		
		// Get mobile numbers.
		
		$numbers = MobileNumber::all();
		
		// Response.
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => count($numbers) ? 200 : 404,
			"results" => $numbers
		]))->getResponseObject());
		
	});
	
	$app->post("/authorised-mobile-numbers", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		// Create mobile number.
		
		$created = MobileNumber::create([
			"title" => $post["title"],
			"mobile_number" => $post["mobile_number"],
		]);

		// die("henry print error " . print_r($created, 1));
		
		// Response.
		
		if ($created) {
			return $response->withJson((new Vinteract\API\Response([ "code" => 201 ]))->getResponseObject());
		}
		
		return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		
	});
	
	$app->post("/authorised-mobile-numbers/{id}", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		// Get mobile number and update it.
		
		$number = MobileNumber::find($args["id"]);
		
		if (isset($number)) {
			
			if (array_key_exists("title", $post)) {
				$number->title = $post["title"];
			}
			
			if (array_key_exists("mobile_number", $post)) {
				$number->mobile_number = $post["mobile_number"];
			}
			
			$updated = $number->save();
			
		}
		
		// Response.
		
		if ($updated) {
			return $response->withJson((new Vinteract\API\Response(["code" => 200]))->getResponseObject());
		}
		
		return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		
	});
	
	$app->delete("/authorised-mobile-numbers/{id}", function(Request $request, Response $response, array $args) {
		
		// Delete mobile number.
		
		$deleted = MobileNumber::destroy($args["id"]);
		
		// Response.
		
		if ($deleted) {
			return $response->withJson((new Vinteract\API\Response(["code" => 200]))->getResponseObject());
		}
		
		return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		
	});
	
	/*
		Send Verification Code
	*/
	
	$app->post("/send-verification-code", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$data = $request->getParsedBody();
		
		// Create verification code.
		
		$code = Vinteract\Auth\VerificationCode::create([
			"scope" => $data["scope"],
			"mobile_number" => $data["mobile_number"],
		]);
		
		// Was the code created successfully? If so, try sending the SMS.
		
		if (isset($code)) {
			
			// Send the SMS.
			
			$sms = new Vinteract\Notifications\SMS();
			
			$smsResponse = $sms->send([
				"mobile_number" => $code->mobile_number,
				"message" => $code->getFormattedMessage(),
			]);
			
			// Response.
			
			return $response->withJson($smsResponse->getResponseObject());
			
		}
		
		// Response.
		
		return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		
	});
	
	/*
		Verify Code
	*/
	
	$app->post("/verify-code", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$data = $request->getParsedBody();
		
		// Get the verification code.
		
		$code = Vinteract\Auth\VerificationCode::where([
			"scope" => $data["scope"],
			"code" => $data["code"],
		])->first();
		
		// Verify that the code was sent to their mobile number.
		
		if (isset($code) and $code->verify()) {
			
			if ($code->shouldSelfDestruct()) {
				$code::destroy($code->id);
			}
			
			return $response->withJson((new Vinteract\API\Response([
				"code" => 200,
				"message" => "Valid verification code.",
			]))->getResponseObject());
			
		}
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => 404,
			"message" => "Invalid verification code.",
		]))->getResponseObject());
		
	});
	
	$app->run();
	