<?php
	
	require("vendor/autoload.php");

	use DateTime as DateTime;
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Client;
	use \Vinteract\Utils;
	use \Vinteract\API\Response as ApiResponse;
	use \Vinteract\Auth\JwtToken;
	use \Vinteract\Encryption\Bcrypt;
	use \Vinteract\Resources\User;
	use \Vinteract\Resources\UserDevice;
	use \Vinteract\Resources\UserGroup;
	use \Vinteract\Resources\UserGroupUser;
	
	use \ReallySimpleJWT\Token;
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Environment
	*/
	
	$whitelist = array(
		"127.0.0.1",
		"::1"
	);
	
	if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
		define("ENVIRONMENT", "development");
	} else {
		define("ENVIRONMENT", "production");
	}
	
	/*
		Middleware
	*/
	
	$headerMiddleware = function($req, $res, $next) {
		
	    $response = $next($req, $res);
		
	    return $response
			->withHeader("Access-Control-Allow-Origin", "*")
			->withHeader("Access-Control-Allow-Headers", "Authorization, X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id, Vinteract-Access-Token")
			->withHeader("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE, PATCH, OPTIONS")
			->withHeader("Access-Control-Expose-Headers", "Authorization");
		
	};
	
	$clientAuthMiddleware = function($request, $response, $next) {
		
		// Init client?
		
		$clientId = $request->getHeader("Vinteract-Client-Id");
		$accessToken = $request->getHeader("Vinteract-Access-Token");
		
		if (count($clientId)) {
			Client::init($clientId[0]);
		}
		
		if (count($accessToken)) {
			$accessToken = (new \Vinteract\Payments\AccessToken())->load();
			if ($accessToken->valid()) {
				Client::init($accessToken->get("public_id"));
			}
		}
		
		// If the access token is valid load the client and boot the payments class.
		
		if (ENVIRONMENT === "development") {
			Client::init("EjNic");
		}
		
		// Return response.
		
		return $next($request, $response);
		
	};
	
	$userAuthMiddleware = function($request, $response, $next) {
		
		$headers = $request->getHeaders();
		
		if (array_key_exists("HTTP_AUTHORIZATION", $headers)) {
			
			$token = Utils::extractBearerToken($request->getHeader("HTTP_AUTHORIZATION")[0]);
			$secret = Client::getJWTSecretKey();
			
			if ($token != null && Token::validate($token, $secret)) {
				$request = $request->withAttribute("user", User::whereUuid(JwtToken::getPayloadData($token, "data.user.uuid"))->first());
			} else {
				return $response->withJson((new ApiResponse([
					"code" => 401,
				]))->getResponseObject(), 401, JSON_NUMERIC_CHECK);
			}
			
		}
		
		return $next($request, $response);
		
	};
	
	$app->add($clientAuthMiddleware);
	$app->add($headerMiddleware);
	
	/*
		Functions
	*/
	
	function getUserGroups(Request $request) {
		
		// Variables.
		
		$userGroups = [];
		
		$post = $request->getParsedBody() ? $request->getParsedBody() : [];
		
		// Check post data for user group preferences.
		
		try {
			
			$preferences = json_decode($post["user_groups"], true);
			
			if (!empty($preferences["country"]) && array_key_exists("country", $preferences)) {
				$userGroups[] = [ "title" => $preferences["country"], "public" => 0 ];
			}
			
			if (!empty($preferences["region"]) && array_key_exists("region", $preferences)) {
				$userGroups[] = [ "title" => $preferences["region"], "public" => 0 ];
			}
			
			if (!empty($preferences["user_groups"]) && array_key_exists("user_groups", $preferences)) {
				foreach ($preferences["user_groups"] as $userGroup) {
					$userGroups[] = [ "title" => $userGroup, "public" => 1 ];
				}
			}
			
		} catch (\Throwable $th) {
			error_log($th);
		}
		
		// Return user groups.
		
		return $userGroups;
		
	}
	
	function getPayload(Request $request) {
		
		// Variables.
		
		$payload = [];
		
		$post = $request->getParsedBody() ? $request->getParsedBody() : [];
		
		$findByKey = "";
		$findByValue = "";
		
		// Check post for user data.
		
		if (array_key_exists("phone", $post) and !empty($post["phone"])) {
			
			$payload["phone"] = $post["phone"];
			
			$findByKey = "phone";
			$findByValue = $post["phone"];
			
		}
		
		if (array_key_exists("email", $post) and !empty($post["email"])) {
			
			$payload["email"] = $post["email"];
			
			$findByKey = "email";
			$findByValue = $post["email"];
			
		}
		
		if (array_key_exists("password", $post) and !empty($post["password"])) {
			$payload["password"] = Bcrypt::hash($post["password"]);
		}
		
		if (array_key_exists("first_name", $post) and !empty($post["first_name"])) {
			$payload["first_name"] = $post["first_name"];
		}
		
		if (array_key_exists("last_name", $post) and !empty($post["last_name"])) {
			$payload["last_name"] = $post["last_name"];
		}
		
		if (array_key_exists("gender", $post) and !empty($post["gender"])) {
			$payload["gender"] = $post["gender"];
		}
		
		if (array_key_exists("dob", $post) and !empty($post["dob"])) {
			$dob = new DateTime($post["dob"]);
			$payload["dob"] = $dob->format("Y-m-d");
		}
		
		if (array_key_exists("street_1", $post) and !empty($post["street_1"])) {
			$payload["street_1"] = $post["street_1"];
		}
		
		if (array_key_exists("street_2", $post) and !empty($post["street_2"])) {
			$payload["street_2"] = $post["street_2"];
		}
		
		if (array_key_exists("country", $post) and !empty($post["country"])) {
			$payload["country"] = $post["country"];
		}
		
		if (array_key_exists("region", $post) and !empty($post["region"])) {
			$payload["region"] = $post["region"];
		}
		
		if (array_key_exists("region_short", $post) and !empty($post["region_short"])) {
			$payload["region_short"] = $post["region_short"];
		}
		
		if (array_key_exists("suburb", $post) and !empty($post["suburb"])) {
			$payload["suburb"] = $post["suburb"];
		}
		
		if (array_key_exists("postcode", $post) and !empty($post["postcode"])) {
			$payload["postcode"] = $post["postcode"];
		}
		
		if (array_key_exists("lat", $post) and !empty($post["lat"])) {
			$payload["lat"] = $post["lat"];
		}
		
		if (array_key_exists("lng", $post) and !empty($post["lng"])) {
			$payload["lng"] = $post["lng"];
		}
		
		if (array_key_exists("app_push_notifications", $post) and !empty($post["app_push_notifications"])) {
			$payload["app_push_notifications"] = $post["app_push_notifications"];
		}
		
		if (array_key_exists("web_push_notifications", $post) and !empty($post["web_push_notifications"])) {
			$payload["web_push_notifications"] = $post["web_push_notifications"];
		}
		
		if (array_key_exists("sms_notifications", $post) and !empty($post["sms_notifications"])) {
			$payload["sms_notifications"] = $post["sms_notifications"];
		}
		
		if (array_key_exists("terms_and_conditions", $post) and !empty($post["terms_and_conditions"])) {
			$payload["terms_and_conditions"] = $post["terms_and_conditions"];
		}
		
		if (array_key_exists("active", $post) and !empty($post["active"])) {
			$payload["active"] = $post["active"];
		}
		
		// Return payload.
		
		return [
			"findByKey" => $findByKey,
			"findByValue" => $findByValue,
			"userData" => $payload,
		];
		
	};
	
	/*
		Login
	*/
	
	$app->post("/login", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody($request);
		
		// If they've provided a username and a password try
		// finding them in the database and validating their password.
		
		if ((isset($post["username"]) and !empty($post["username"])) and (isset($post["password"]) and !empty($post["password"]))) {
			
			$user = \Vinteract\Resources\User::where("email", $post["username"])
				->orWhere("phone", $post["username"])
				->first();
			
			if ((isset($user) and $user->active) and Bcrypt::compare($post["password"], $user->password)) {
				
				$jwt = (new \Vinteract\Auth\JwtToken())
					->setIat(time())
					->setData(["user" => $user])
					->generate();
				
				return $response
					->withHeader("Authorization", $jwt->getToken())
					->withJson((new ApiResponse([
						"code" => 200,
						"contents" => [ "user" => $user ]
					]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
				
			}
			
		}
		
		// Default error response.
		
		return $response
			->withJson((new ApiResponse([
				"message" => "Please check your login credentials and try again."
			]))->getResponseObject(), 400, JSON_NUMERIC_CHECK);
		
	});
	
	/*
		Create Account
	*/
	
	$app->post("/create-account", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$payload = getPayload($request);
		
		$emailAddressUsed = false;
		$phoneNumberUsed = false;
		
		if (array_key_exists("email", $payload["userData"])) {
			$emailAddressUsed = User::where("email", $payload["userData"]["email"])->exists();
		}
		
		if (array_key_exists("phone", $payload["userData"])) {
			$phoneNumberUsed = User::where("phone", $payload["userData"]["phone"])->exists();
		}
		
		// Check if the user exists or not.
		
		if ($emailAddressUsed or $phoneNumberUsed) {
			
			if ($payload["findByKey"] === "phone") {
				$message = "This phone number has already been used to create an account.";
			} else {
				$message = "This email address has already been used to create an account.";
			}
			
			return $response->withJson((new ApiResponse([
				"code" => 400,
				"message" => $message,
			]))->getResponseObject(), 400, JSON_NUMERIC_CHECK);
			
		} else {
			
			$newUser = User::create($payload["userData"]);
			
			if ($newUser) {
				
				// Variables.
				
				$successMessage = "Success! Your account has been created.";
				
				$hasMobileNumber = isset($newUser->phone);
				$hasEmailAddress = isset($newUser->email);
				
				$sendAuthenticationEmail = $hasEmailAddress and !$newUser->active;
				// $sendAuthenticationEmail = ($hasEmailAddress and !$hasMobileNumber) and !$newUser->active;
				
				if ($sendAuthenticationEmail) {
					$successMessage = "Success! Your account has been created. Check your email account for activation instructions.";
				}
				
				// Create user groups?
				
				$userGroups = getUserGroups($request);
				
				foreach($userGroups as $userGroup) {
					
					$newUserGroup = UserGroup::where([
						["title", "=", $userGroup["title"]],
						["public", "=", $userGroup["public"]],
					])->get();
					
					if ($newUserGroup->isEmpty()) {
						$newUserGroup = UserGroup::create($userGroup);
					} else {
						$newUserGroup = $newUserGroup->first();
					}
					
					$userGroupUser = UserGroupUser::create([
						"user_id" => $newUser->id,
						"user_group_id" => $newUserGroup->id
					]);
					
				}
				
				// Send authentication email?
				// Only if they're got an email address
				// assigned to their account.
				
				if ($sendAuthenticationEmail) {
					
					$authenticationCode = \Vinteract\Resources\AuthenticationCode::create([ "content_type" => "users", "content_id" => $newUser->id ]);
					
					$emailClient = new \Vinteract\Notifications\Email();
					
					$emailTemplate = \Vinteract\Notifications\EmailTemplates::getTemplate("newUserAuthentication", [
						"{CLIENT_NAME}" => Client::getDisplayName(),
						"{AUTHENTICATION_URL}" => $authenticationCode->getURL(),
					]);
					
					$emailResponse = $emailClient->send([
						"to" => [ [ "name" => $newUser->name, "email" => $newUser->email ] ],
						"subject" => $emailTemplate["subject"],
						"html" => $emailTemplate["html"],
					]);
					
				}
				
				// Create JWT token.
				
				$jwtToken = Token::customPayload([
					"iat" => time(),
					"exp" => (time() + (60 * 60)),
					"iss" => "localhost",
					"data" => [ "user" => $newUser ]
				], Client::getJWTSecretKey());
				
				// Response.
				
				return $response
					->withHeader("Authorization", $jwtToken)
					->withJson((new ApiResponse([
						"code" => 201,
						"message" => $successMessage,
						"contents" => [ "user" => $newUser ]
					]))->getResponseObject(), 201, JSON_NUMERIC_CHECK);
				
			}
			
		}
		
		return $response->withJson((new ApiResponse())->getResponseObject(), 400, JSON_NUMERIC_CHECK);
		
	});
	
	/*
		Update Account
	*/
	
	$app->post("/update/{uuid}", function(Request $request, Response $response, array $args) {
		
		$payload = getPayload($request);
		
		$user = User::whereUuid($args["uuid"])->first();
		
		if (isset($user)) {
			
			// Update user details.
			
			foreach ($payload["userData"] as $key => $value) {
				$user->$key = $value;
			}
			
			$user->save();
			
			// Create JWT token.
			
			$jwtToken = Token::customPayload([
				"iat" => time(),
				"exp" => (time() + (60 * 60)),
				"iss" => "localhost",
				"data" => [ "user" => $user ]
			], Client::getJWTSecretKey());
			
			// Response.
			
			return $response
				->withHeader("Authorization", $jwtToken)
				->withJson((new ApiResponse([
					"code" => 200,
					"contents" => [ "user" => $user ]
				]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
			
		}
		
		// Response.
		
		return $response->withJson((new ApiResponse(["code" => 404]))->getResponseObject(), 404, JSON_NUMERIC_CHECK);
		
	})->add($userAuthMiddleware);
	
	/*
		Update Location
	*/
	
	$app->post("/update-location/{uuid}", function(Request $request, Response $response, array $args) {
		
		$payload = getPayload($request);
		
		$user = User::whereUuid($args["uuid"])->first();
		
		if (isset($user) and array_key_exists("lat", $payload["userData"]) and array_key_exists("lng", $payload["userData"])) {
			
			// Update user location.
			
			$user->lat = $payload["userData"]["lat"];
			$user->lng = $payload["userData"]["lng"];
			
			$user->save();
			
			// Response.
			
			return $response->withJson((new ApiResponse(["code" => 200]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
			
		}
		
		// Response.
		
		return $response->withJson((new ApiResponse(["code" => 400]))->getResponseObject(), 400, JSON_NUMERIC_CHECK);
		
	});
	
	/*
		Device
	*/
	
	$app->post("/device", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$user = $request->getAttribute("user");
		
		// Lookup the device and if it doesn't exist create it.
		
		$userDevice = UserDevice::where([
			["user_id", "=", $user->id],
			["device_id", "=", $post["token"]],
		])->get();
		
		if ($userDevice->isEmpty()) {
			
			$newUserDevice = UserDevice::create([
				"user_id" => $user->id,
				"device_id" => $post["token"],
				"platform" => $post["platform"],
			]);
			
			foreach($post["meta"] as $key => $value) {
				Client::local("users_devices_meta")->updateOrInsert(
					["device_id" => $newUserDevice->id, "key" => $key],
					["key" => $key, "value" => $value]
				);
			}
			
			if ($newUserDevice) {
				return $response->withJson((new ApiResponse(["code" => 201]))->getResponseObject(), 201, JSON_NUMERIC_CHECK);
			}
			
		} else {
			
			foreach($post["meta"] as $key => $value) {
				Client::local("users_devices_meta")->updateOrInsert(
					["device_id" => $userDevice->first()->id, "key" => $key],
					["key" => $key, "value" => $value]
				);
			}
			
		}
		
		return $response->withJson((new ApiResponse(["code" => 200]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
		
	})->add($userAuthMiddleware);
	
	$app->delete("/device/{uuid}", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$user = $request->getAttribute("user");
		
		// Lookup the user group and if it exists
		// remove the user from it.
		
		$userDevice = UserDevice::whereUuid($args["uuid"])->get();
		
		if (!$userDevice->isEmpty()) {
			$deleted = UserDevice::whereUserId($user->id)->whereUserGroupId($userDevice->first()->id)->delete();
			if ($deleted) {
				return $response->withJson((new ApiResponse(["code" => 200]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
			}
		}
		
		return $response->withJson((new ApiResponse())->getResponseObject(), 400, JSON_NUMERIC_CHECK);
		
	})->add($userAuthMiddleware);
	
	/*
		Saved Credit Cards
	*/
	
	$app->get("/saved-cards", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		// Get user's cards?
		
		if ($request->getAttribute("user")) {
			$user = $request->getAttribute("user");
			$cards = \Vinteract\Payments\CardTokenRecord::whereUserId($user->id)->get();
			if (count($cards)) {
				return $response->withJson((new ApiResponse([
					"code" => 200,
					"results" => $cards,
				]))->getResponseObject(), 200, JSON_NUMERIC_CHECK);
			}
		}
		
		// Response.
		
		return $response->withJson((new ApiResponse(["code" => 404]))->getResponseObject(), 404, JSON_NUMERIC_CHECK);
		
	})->add($userAuthMiddleware);
	
	/*
		
		Catch-all route to serve a 404 Not Found page if none of the routes match
		NOTE: make sure this route is defined last
		
	*/
	
	$app->map(["GET", "POST", "PUT", "DELETE", "PATCH"], "/{routes:.+}", function($req, $res) {
	    $handler = $this->notFoundHandler; // handle using the default Slim page not found handler
	    return $handler($req, $res);
	});
	
	/*
		Run App
	*/
	
	$app->run();
	
