<?php
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Client as Client;
	
	require("vendor/autoload.php");
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Environment
	*/
	
	$whitelist = array(
		"127.0.0.1",
		"::1"
	);

	if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
		define("ENVIRONMENT", "development");
	} else {
		define("ENVIRONMENT", "production");
	}
	
	/*
		Middleware
	*/
	
	$app->add(function ($req, $res, $next) {
	    $response = $next($req, $res);
	    return $response
			->withHeader('Access-Control-Allow-Origin', '*')
			->withHeader('Access-Control-Allow-Headers', 'X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id, Vinteract-Token')
			->withHeader('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, PATCH, OPTIONS');
	});
	
	$app->add(function($request, $response, $next) {
		
		// Init client?
		
		$headers = $request->getHeader("Vinteract-Client-Id");
		
		if (count($headers)) {
			Vinteract\Client::init($headers[0]);

			$token = $request->getHeader("Vinteract-Token");

			if (empty ($token)) {
				return $response->withStatus(401);
			}

			$valid = Vinteract\Auth\JwtToken::validate($token[0]);

			if (! $valid) {
				return $response->withStatus(401);
			}
		}
		
		// Return response.
		
		return $next($request, $response);
		
	});
	
	/*
		Announcements
	*/
	
	$app->get("/announcements", function(Request $request, Response $response, array $args) {
		
		$results = \Vinteract\CMS\DashboardAnnouncement::all();
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => count($results) ? 200 : 404,
			"results" => isset($results) ? $results : null,
		]))->getResponseObject());
		
	});
	
	/*
		Payments
	*/
	
	$app->get("/payments/init", function(Request $request, Response $response, array $args) {
		
		// Boot payments.
		
		\Vinteract\Payments\Payments::boot();
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"enabled" => \Vinteract\Features::isFeatureEnabled("vinteract_pay"),
				"account" => \Vinteract\Payments\Payments::getAccountDetails([
					"information" => [ "legal_entity_name", "trading_name" ],
				]),
				"platform" => \Vinteract\Payments\Payments::getPlatformDetails(),
				"client_type" => Client::getType(),
			]
		]))->getResponseObject());
		
	});
	
	$app->get("/payments/transactions", function(Request $request, Response $response, array $args) {
		
		// Boot payments.
		
		\Vinteract\Payments\Payments::boot();
		
		// Get transactions.
		
		$results = \Vinteract\Payments\TransactionRecord::fetch(null, $request->getQueryParams());
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => count($results) ? 200 : 404,
			"results" => isset($results) ? $results : null,
		]))->getResponseObject());
		
	});
	
	/*
		Analytics
	*/
	
	$app->get("/analytics", function(Request $request, Response $response, array $args) {
		
		$results = \Vinteract\Analytics\Event::fetch($request->getQueryParams());
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => count($results) ? 200 : 404,
			"results" => isset($results) ? $results : null,
		]))->getResponseObject());
		
	});
	
	$app->get("/analytics/init", function(Request $request, Response $response, array $args) {
		
		// Settings.
		
		$settings = [
			"colours" => [ "ios" => "#3BABDA", "android" => "#A5C445", "web" => "#999999" ],
		];
		
		$settings["client_type"] = Client::getType();
		
		$settings["labels"] = array_column(Client::local("settings_labels")->get()->toArray(), "value", "key");
		
		$settings["features"] = array_column(Client::local("settings_features")->get()->toArray(), "value", "key");
		
		$settings["allowed_events"] = [
			"view_home_screen" => [ "label_key" => "home", "permission_key" => "content_home" ],
			"view_gallery" => [ "label_key" => "gallery", "permission_key" => "content_gallery" ],
			"view_blog_url" => [ "label_key" => "blog", "permission_key" => "us_blog" ],
			"view_giving_url" => [ "label_key" => "giving", "permission_key" => "us_giving" ],
			"view_shop_url" => [ "label_key" => "shop", "permission_key" => "us_shop" ],
			"view_content_asa_programs" => [ "label_key" => "asa_programs", "permission_key" => "content_asa_programs" ],
			"view_content_verses" => [ "label_key" => "verses", "permission_key" => "content_verses" ],
			"view_content_devotionals" => [ "label_key" => "devotionals", "permission_key" => "content_devotionals" ],
			"view_content_advice" => [ "label_key" => "advice", "permission_key" => "content_advice" ],
			"view_content_news" => [ "label_key" => "news", "permission_key" => "content_news" ],
			"view_content_newsletters" => [ "label_key" => "newsletters", "permission_key" => "content_newsletters" ],
			"view_content_promotional_content" => [ "label_key" => "promotional_content", "permission_key" => "content_promotional_content" ],
			"view_content_sermons" => [ "label_key" => "sermons", "permission_key" => "content_sermons" ],
			"view_content_events" => [ "label_key" => "events", "permission_key" => "content_events" ],
			"view_content_fan_wall" => [ "label_key" => "fan_wall", "permission_key" => "content_fan_wall" ],
			"view_content_prayer_requests" => [ "label_key" => "prayer_requests", "permission_key" => "content_prayer_requests" ],
			"view_content_offers" => [ "label_key" => "offers", "permission_key" => "content_offers" ],
			"view_content_about" => [ "label_key" => "about", "permission_key" => "us_about" ],
			"view_content_services" => [ "label_key" => "services", "permission_key" => "us_services" ],
			"view_content_ministries" => [ "label_key" => "ministries", "permission_key" => "us_ministries" ],
			"view_content_team" => [ "label_key" => "team", "permission_key" => "us_team" ],
			"view_content_locations" => [ "label_key" => "locations1", "permission_key" => "us_locations" ],
			"view_content_church_locations" => [ "label_key" => "locations1", "permission_key" => "us_church_locations" ],
		];
		
		// Program categories?
		
		if (Client::getType() === "businesses") {
			
			$program_categories = Client::local("saved_categories_asa_programs")->get()->toArray();
			
			$settings["program_categories"] = array_reduce($program_categories, function($result, $item) {
				$item = (array) $item;
				$result[$item["id"]] = $item;
				return $result;
			}, []);
			
		}
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [ "settings" => $settings ],
		]))->getResponseObject());
		
	});
	
	/*
		Data Metre
	*/
	
	$app->get("/data-metre/years", function(Request $request, Response $response, array $args) {
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"min" => intval(Client::local("data_usage")->min("year")),
				"max" => intval(Client::local("data_usage")->max("year")),
			],
		]))->getResponseObject());
		
	});
	
	$app->get("/data-metre/months", function(Request $request, Response $response, array $args) {
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"min" => intval(Client::local("data_usage")->where("year", $request->getQueryParam("year"))->min("month")),
				"max" => intval(Client::local("data_usage")->where("year", $request->getQueryParam("year"))->max("month")),
			],
		]))->getResponseObject());
		
	});
	
	$app->get("/data-metre/days", function(Request $request, Response $response, array $args) {
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"min" => intval(Client::local("data_usage")->where("month", $request->getQueryParam("month"))->where("year", $request->getQueryParam("year"))->min("day")),
				"max" => intval(Client::local("data_usage")->where("month", $request->getQueryParam("month"))->where("year", $request->getQueryParam("year"))->max("day")),
			],
		]))->getResponseObject());
		
	});
	
	$app->get("/data-metre/usage", function(Request $request, Response $response, array $args) {
		
		$day = $request->getQueryParam("day", 0);
		$month = $request->getQueryParam("month", 0);
		$year = $request->getQueryParam("year", 0);
		
		// Get "Up" data use.
		
		$up = Client::local("data_usage");
		
		if ($request->getQueryParam("day")) {
			$up->where("day", $day);
		}
		
		if ($request->getQueryParam("month")) {
			$up->where("month", $month);
		}
		
		if ($request->getQueryParam("year")) {
			$up->where("year", $year);
		}
		
		$up = $up->sum("up");
		
		// Get "Down" data use.
		
		$down = Client::local("data_usage");
		
		if ($request->getQueryParam("day")) {
			$down->where("day", $day);
		}
		
		if ($request->getQueryParam("month")) {
			$down->where("month", $month);
		}
		
		if ($request->getQueryParam("year")) {
			$down->where("year", $year);
		}
		
		$down = $down->sum("down");
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"up" => intval($up),
				"down" => intval($down),
				"day" => intval($day),
				"month" => intval($month),
				"year" => intval($year),
			],
		]))->getResponseObject());
		
	});
	
	$app->get("/data-metre/quota", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$quota_id = Client::getData("data_quota");
		$quota_custom = Client::getData("data_quota_custom") === "true";
		
		$quota_up = 0;
		$quota_down = 0;
		
		// Get quota limits.
		
		if ($quota_custom) {
			$quota = Client::global("global_" . Client::getType() . "_platform_quotas_custom")->find($quota_id);
		} else {
			$quota = Client::global("global_" . Client::getType() . "_platform_quotas")->find($quota_id);
		}
		
		$quota_up = ($quota->up * 1024 * 1024 * 1024);
		$quota_down = ($quota->down * 1024 * 1024 * 1024);
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"up" => $quota_up,
				"down" => $quota_down,
			],
		]))->getResponseObject());
		
	});
	
	/*
		File Storage
	*/
	
	$app->get("/file-storage", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$org_name = Client::getName();
		$org_type = Client::getType();
		
		$uploads_path = \Vinteract\Utils::getCmsPath("uploads/{$org_type}/{$org_name}");
		
		// Get files.
		
		$images = \Vinteract\Utils::getFolderContentsAndSize("{$uploads_path}/images");
		
		$documents = \Vinteract\Utils::getFolderContentsAndSize("{$uploads_path}/documents");
		
		$audio = \Vinteract\Utils::getFolderContentsAndSize("{$uploads_path}/audio");
		
		// Calculate storage space totals.
		
		$total_storage_space = intval(Client::getData("disk_space") * 1024 * 1024 * 1024);
		
		$used_storage_space = ($images["size"] + $documents["size"] + $audio["size"]);
		
		$available_storage_space = ($total_storage_space - $used_storage_space);
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"files" => [
					"count" => $images["count"] + $documents["count"] + $audio["count"],
					"size" => $images["size"] + $documents["size"] + $audio["size"], 
				],
				"storage" => [
					"total" => $total_storage_space,
					"available" => $available_storage_space,
					"used" => $used_storage_space,
				],
			],
		]))->getResponseObject());
		
	});
	
	/*
		Run App
	*/
	
	$app->run();
	
