<?php
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Client;
	
	use \Minishlink\WebPush\WebPush;
	use \Minishlink\WebPush\Subscription;
	use \Minishlink\WebPush\VAPID;
	
	use \libphonenumber\PhoneNumberUtil;
	use \libphonenumber\PhoneNumberFormat;
	
	use DateTime as DateTime;
	use DateTimezone as DateTimezone;
	
	require("vendor/autoload.php");
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Constants
	*/
	
	define("SMS_OPT_OUT_MESSAGE", "SMS Stop to OptOut");
	
	/*
		Environment
	*/
	
	// $whitelist = array(
	// 	"127.0.0.1",
	// 	"::1"
	// );
	
	// if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
	// 	define("ENVIRONMENT", "development");
	// } else {
		define("ENVIRONMENT", "production");
	// }
	
	/*
		Middleware
	*/
	
	$headerMiddleware = function($req, $res, $next) {
		
	    $response = $next($req, $res);
		
	    return $response
			->withHeader("Access-Control-Allow-Origin", "*")
			->withHeader("Access-Control-Allow-Headers", "X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id")
			->withHeader("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE, PATCH, OPTIONS");
		
	};
	
	$authMiddleware = function($request, $response, $next) {
		
		// Init client?
		
		$headers = $request->getHeader("Vinteract-Client-Id");
		
		if (count($headers)) {
			Vinteract\Client::init($headers[0]);
		}
		
		if (ENVIRONMENT === "development") {
			Vinteract\Client::init("EjNic");
		}
		
		// Return response.
		
		return $next($request, $response);
		
	};
	
	$app->add($headerMiddleware);
	
	/*
		Init
	*/
	
	$app->get("/init", function(Request $request, Response $response, array $args) {
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"features" => \Vinteract\Features::getAllFeatures(),
				"sms_opt_out_message" => SMS_OPT_OUT_MESSAGE,
			],
		]))->getResponseObject());
		
	})->add($authMiddleware);
	
	/*
		Check Queued Notifications
	*/
	
	$sendNotification = function(\Vinteract\Resources\QueuedNotification $notification) {
		
		if ($notification->readyToSend()) {
			
			// Variables.
			
			$orgName = \Vinteract\Client::getDisplayName();
			$orgPublicId = \Vinteract\Client::getPublicId();
			$orgDbPrefix = \Vinteract\Client::getData("title");
			
			$contentType = $notification->content_type;
			$contentId = $notification->content_id;
			
			if ($notification->content_type === "custom") {
				$body = $notification->body;
				$contentId = null;
			} else {
				$body = $notification->content_item->body;
				$contentId = $notification->content_item->id;
			}
			
			$to = $notification->to;
			
			// Trim body?
			
			$length = 117;
			
			if ($contentType != "custom") {
				if (strlen($body) > $length) {
					$body = substr($body, 0, $length) . "...";
				}
			}
			
			// Send app push?
			
			if ($notification->app) {
				
				$customApp = \Vinteract\Client::getData("custom_app") == "1";
				$churchCentralApp = \Vinteract\Client::getData("church_central") == "1";
				
				$pushClientOptions = [];
				
				$pushPayload = [
					"alert" => $body,
					"alert_original" => $body,
					"sound" => "default",
					"badge" => "+1",
					"org_name" => $orgName,
					"org_public_id" => $orgPublicId,
				];
				
				if ($contentType != "custom") {
					$pushPayload["content_type"] = $contentType;
					$pushPayload["content_id"] = $contentId;
				}
				
				// If the notification is going to a user group / individuals find the device tokens
				// and add them to the push client options so that we target the correct users.
				
				if ($to === "user_group") {
					
					$userGroups = \Vinteract\Resources\UserGroup::with(["users"])->find(json_decode($notification->user_group, true));
					$userIds = array_column(\Vinteract\Utils::mergeArrayOfArrays($userGroups->pluck("users")->toArray()), "id");
					
					$users = \Vinteract\Resources\User::with(["devices"])->hasAppPushSubscriptions()->find($userIds);
					$devices = \Vinteract\Utils::mergeArrayOfArrays($users->pluck("devices")->toArray());
					
					$pushClientOptions["tokens"] = join(",", array_column($devices, "device_id"));
					
				} elseif ($to === "individuals") {
					
					$users = \Vinteract\Resources\User::with(["devices"])->hasAppPushSubscriptions()->find(explode(",", $notification->individuals));
					
					$devices = \Vinteract\Utils::mergeArrayOfArrays($users->pluck("devices")->toArray());
					
					$pushClientOptions["tokens"] = join(",", array_column($devices, "device_id"));
					
				}
				
				// Send the push notification.
				
				$pushClient = new \Vinteract\Notifications\AppPush();
				
				if ($customApp) {
					
					$pushClientOptions["api_key"] = \Vinteract\Client::getAPIKey("push_api_key");
					$pushClientOptions["channel"] = "alert";
					$pushClientOptions["payload"] = $pushPayload;
					
					$pushResponse = $pushClient->send($pushClientOptions);
					
				}
				
				if ($churchCentralApp) {
					
					$pushPayload["alert"] = "{$orgName}: {$body}";
					
					$pushClientOptions["api_key"] = \Vinteract\Client::getChurchCentralPushAPIKey();
					$pushClientOptions["channel"] = $orgDbPrefix;
					$pushClientOptions["payload"] = $pushPayload;
					
					$pushResponse = $pushClient->send($pushClientOptions);
					
				}
				
			}
			
			// Send web push?
			
			if ($notification->web) {
				
				// Push payload.
				
				$webPushPayload = json_encode([
					"title" => $orgName,
					"message" => $body,
					"url" => Client::getWebsiteURL(),
				]);
				
				// Get users.
				
				if ($to === "user_group" or $to === "individuals") {
					$users = \Vinteract\Resources\User::hasWebPushSubscriptions()->get($notification->individuals);
				} else {
					$users = \Vinteract\Resources\User::hasWebPushSubscriptions()->get();
				}
				
				// Instantiate web push client.
				
				$webPushClient = new WebPush([
					"VAPID" => [
						"subject" => $orgName,
						"publicKey" => Client::getAPIKey("web_push_public_vapid_key"),
						"privateKey" => Client::getAPIKey("web_push_private_vapid_key"),
					],
				], [], 30, ["verify" => false]);
				
				// Create the subscriptions and send the notifications.
				
				foreach ($users as $user) {
					foreach($user->devices as $device) {
						$subscriptions[] = Subscription::create([
							"endpoint" => $device["token"],
							"publicKey" => $device["meta"]["public_key"],
							"authToken" => $device["meta"]["auth_token"],
							"contentEncoding" => "aesgcm",
						]);
					}
				}
				
				// Send the notifications.
				
				foreach($subscriptions as $subscription) {
					$webPushClient->sendNotification($subscription, $webPushPayload, true);
				}
				
				// Flush the notifications and store the response.
				
				foreach ($webPushClient->flush() as $report) {
					if ($report->isSuccess()) {
						//$responses[] = [ "success" => true, "error" => false, "message" => "Success" ];
					} else {
						//$responses[] = [ "success" => false, "error" => true, "message" => $report->getReason() ];
					}
				}
				
			}
			
			// Send SMS?
			
			if ($notification->sms) {
				
				
				
			}
			
			// Disable the notification until next time.
			
			if ($notification->isCustomDateNotification()) {
				$notification->disableCustomDate();
			} else {
				$notification->moveSendDateForward();
			}
			
			// Store the sent notification in the database.
			
			$created = \Vinteract\Resources\SentNotification::create([
				"body" => $body,
				"content_type" => $notification->content_type === "custom" ? "Notification" : $notification->content_type,
				"content_id" => $contentId,
				"to_who" => $notification->to,
				"user_group" => $notification->user_group,
				"organisation" => $notification->organisation,
				"individuals" => $notification->individuals,
				"app" => $notification->app,
				"web" => $notification->web,
				"sms" => $notification->sms,
			]);
			
		}
		
	};
	
	$app->get("/check-queued", function(Request $request, Response $response, array $args) use ($sendNotification) {
		
		// Variables
		
		$db = new \Vinteract\DB(Client::getDatabaseConnections());
		
		$global = $db->capsule->getConnection("default");
		
		/*
			Send church notifications
		*/
		
		$churches = $global->table("global_churches")->get();
		
		foreach ($churches as $church) {
			
			\Vinteract\Client::init($church->public_id);
			
			$notifications = \Vinteract\Resources\QueuedNotification::all();
			
			foreach($notifications as $notification) {
				$sendNotification($notification);
			}
			
		}
		
		/*
			Send business notifications
		*/
		
		$businesses = $global->table("global_businesses")->get();
		
		foreach ($businesses as $business) {
			
			\Vinteract\Client::init($business->public_id);
			
			$notifications = \Vinteract\Resources\QueuedNotification::all();
			
			foreach($notifications as $notification) {
				$sendNotification($notification);
			}
			
		}

		return $response;
		
	});
	
	/*
		Users
	*/
	
	$app->get("/users", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$devices_table = (new \Vinteract\Resources\UserDevice)->getTable();
		
		$user_groups_table = (new \Vinteract\Resources\UserGroup)->getTable();
		
		$platforms = ["Android", "android", "iOS", "ios"];
		
		// Start user query.
		
		$users = \Vinteract\Resources\User::with(["user_groups", "devices"])->orderBy("first_name");
		
		// Apply filters.
		
		if ($request->getQueryParam("app") or $request->getQueryParam("web")) {
			$users->hasPushSubscriptions($request->getQueryParam("app"), $request->getQueryParam("web"));
		}
		
		if ($request->getQueryParam("sms")) {
			$users->hasMobileNumber();
		}
		
		if ($request->getQueryParam("user_groups")) {
			$users->whereHas("user_groups", function($query) use ($user_groups_table, $request) {
				$query->whereIn("{$user_groups_table}.id", $request->getQueryParam("user_groups"));
			});
		}
		
		if ($request->getQueryParam("closest_to_location") and $request->getQueryParam("distance_limit")) {
			
			// Variables.
			
			$latAndLng = explode(",", $request->getQueryParam("closest_to_location"));
			$distanceLimit = $request->getQueryParam("distance_limit");
			
			$lat = $latAndLng[0];
			$lng = $latAndLng[1];
			
			// Expose distance as a column. 
			
			$users->addSelect(\Vinteract\Client::db()->raw("(
					6371 * acos (
					cos (radians({$lat}))
					* cos(radians(lat))
					* cos(radians(lng) - radians($lng))
					+ sin(radians($lat))
					* sin(radians(lat))
				)
			) AS distance"));
			
			// Filter users by the distance limit.
			
			$users->whereRaw("(
					6371 * acos (
					cos (radians({$lat}))
					* cos(radians(lat))
					* cos(radians(lng) - radians($lng))
					+ sin(radians($lat))
					* sin(radians(lat))
				)
			) <= {$distanceLimit}");
			
		}
		
		// Get users.
		
		$users = $users->get();
		
		// Response.
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => count($users) ? 200 : 404,
			"results" => count($users) ? $users : null,
		]))->getResponseObject());
		
	})->add($authMiddleware);
	
	/*
		Sent Notifications
	*/
	
	$sentCallback = function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$perPage = 8;
		
		// Get specific notification or all?
		
		if (array_key_exists("id", $args)) {
			$pagination = [];
			$notifications = \Vinteract\Resources\SentNotification::whereId($args["id"])->get();
		} else {
			$pagination = \Vinteract\Resources\SentNotification::paginate($perPage, ["*"], "page", $request->getQueryParam("page") ? $request->getQueryParam("page") : 1);
			$notifications = $pagination->getCollection();
		}
		
		// Response.
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => count($notifications) ? 200 : 404,
			"results" => count($notifications) ? $notifications : null,
			"contents" => [
				"pagination" => $pagination,
			]
		]))->getResponseObject());
		
	};
	
	$app->get("/sent", $sentCallback)->add($authMiddleware);
	
	$app->get("/sent/{id}", $sentCallback)->add($authMiddleware);
	
	$app->post("/sent", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$notification = $post["notification"];
		
		$type = $notification["content_type"];
		
		$ug = $notification["user_groups"];
		
		$ind = $notification["individuals"];
		
		// Create the sent notification.
		
		$created = \Vinteract\Resources\SentNotification::create([
			"body" => $notification["body"],
			"content_type" => $type === "custom" ? "Notification" : $notification["content_type"],
			"content_id" => $type === "custom" ? null : $notification["content_id"],
			"to_who" => $notification["to"],
			"user_group" => count($ug) ? json_encode($ug) : null,
			"individuals" => count($ind) ? implode(",", $ind) : null,
			"manual" => 1,
			"app" => $notification["channels"]["app"],
			"web" => $notification["channels"]["web"],
			"sms" => $notification["channels"]["sms"],
		]);
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 201,
		]))->getResponseObject());
		
	})->add($authMiddleware);
	
	$app->delete("/sent", function(Request $request, Response $response, array $args) {
		
		$deleted = \Vinteract\Resources\SentNotification::destroy($request->getQueryParam("selected"));
		
		if ($deleted) {
			return $response->withJson((new Vinteract\API\Response([ "code" => 200 ]))->getResponseObject());
		} else {
			return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		}
		
	})->add($authMiddleware);
	
	/*
		Queued Notifications
	*/
	
	$getQueuedCallback = function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$perPage = 8;
		
		// Get specific notification or all?
		
		if (array_key_exists("id", $args)) {
			$pagination = [];
			$notifications = \Vinteract\Resources\QueuedNotification::whereId($args["id"])->get();
		} else {
			$pagination = \Vinteract\Resources\QueuedNotification::orderByDesc("id")->paginate($perPage, ["*"], "page", $request->getQueryParam("page") ? $request->getQueryParam("page") : 1);
			$notifications = $pagination->getCollection();
		}
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => count($notifications) ? 200 : 404,
			"results" => count($notifications) ? $notifications : null,
			"contents" => [
				"pagination" => $pagination,
			]
		]))->getResponseObject());
		
	};
	
	$postQueuedCallback = function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$notification = $post["notification"];
		
		$type = $notification["content_type"];
		
		$ug = $notification["user_groups"];
		
		$ind = $notification["individuals"];
		
		$cd = $notification["custom_dates"];
		
		usort($cd, function($a, $b) {
			return strtotime($a["day"]) - strtotime($b["day"]);
		});
		
		$startDate = new DateTime($notification["start_date"]);
		
		$startDateOffset = clone $startDate;
		$startDateOffset->setTimezone(new DateTimezone(Client::getTimezone()));
		
		// Create or updated the queued notification.
		
		if (array_key_exists("id", $args)) {
			
			$queued = \Vinteract\Resources\QueuedNotification::find($args["id"]);
			
			$queued->title = $notification["title"];
			$queued->body = $notification["body"];
			$queued->content_type = $type;
			$queued->content_id = $type === "custom" ? null : $notification["content_id"];
			$queued->to = $notification["to"];
			$queued->user_group = count($ug) ? json_encode($ug) : null;
			$queued->individuals = count($ind) ? implode(",", $ind) : null;
			$queued->occurrence = $notification["occurrence"];
			$queued->start_date = $startDate->format("Y-m-d H:i:s");
			$queued->send_date = $startDate->format("Y-m-d H:i:s");
			$queued->time = $startDateOffset->format("g:i A");
			$queued->custom_dates = count($cd) ? $cd : null;
			$queued->app = $notification["channels"]["app"];
			$queued->web = $notification["channels"]["web"];
			$queued->sms = $notification["channels"]["sms"];
			
			$updated = $queued->save();
			
			if ($updated) {
				return $response->withJson((new \Vinteract\API\Response([
					"code" => 200,
					"contents" => [
						"updated" => $updated,
					]
				]))->getResponseObject());
			}
			
		} else {
			
			$created = \Vinteract\Resources\QueuedNotification::create([
				"title" => $notification["title"],
				"body" => $notification["body"],
				"content_type" => $type,
				"content_id" => $type === "custom" ? null : $notification["content_id"],
				"to" => $notification["to"],
				"user_group" => count($ug) ? json_encode($ug) : null,
				"individuals" => count($ind) ? implode(",", $ind) : null,
				"occurrence" => $notification["occurrence"],
				"start_date" => $startDate->format("Y-m-d H:i:s"),
				"send_date" => $startDate->format("Y-m-d H:i:s"),
				"time" => $startDateOffset->format("g:i A"),
				"custom_dates" => count($cd) ? $cd : null,
				"app" => $notification["channels"]["app"],
				"web" => $notification["channels"]["web"],
				"sms" => $notification["channels"]["sms"],
			]);
			
			if ($created) {
				return $response->withJson((new \Vinteract\API\Response([
					"code" => 201,
					"contents" => [
						"created" => $created,
					]
				]))->getResponseObject());
			}
			
		}
		
		// Default response.
		
		return $response->withJson((new \Vinteract\API\Response())->getResponseObject());
		
	};
	
	$app->get("/queued", $getQueuedCallback)->add($authMiddleware);
	
	$app->get("/queued/{id}", $getQueuedCallback)->add($authMiddleware);
	
	$app->post("/queued", $postQueuedCallback)->add($authMiddleware);
	
	$app->post("/queued/{id}", $postQueuedCallback)->add($authMiddleware);
	
	$app->delete("/queued", function(Request $request, Response $response, array $args) {
		
		$deleted = \Vinteract\Resources\QueuedNotification::destroy($request->getQueryParam("selected"));
		
		if ($deleted) {
			return $response->withJson((new Vinteract\API\Response([ "code" => 200 ]))->getResponseObject());
		} else {
			return $response->withJson((new Vinteract\API\Response())->getResponseObject());
		}
		
	})->add($authMiddleware);
	
	/*
		Send Notification
	*/
	
	$app->post("/send/app", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$notification = $post["notification"];
		
		$type = $notification["content_type"];
		
		$to = $notification["to"];
		
		$orgName = \Vinteract\Client::getDisplayName();
		$orgPublicId = \Vinteract\Client::getPublicId();
		$orgDbPrefix = \Vinteract\Client::getData("title");
		
		$customApp = \Vinteract\Client::getData("custom_app") == "1";
		$churchCentralApp = \Vinteract\Client::getData("church_central") == "1";
		
		$pushClientOptions = [];
		
		$body = $notification["body"];
		
		// Push notification payload.
		
		$pushPayload = [
			"alert" => $body,
			"alert_original" => $body,
			"sound" => "default",
			"badge" => "+1",
			"org_name" => $orgName,
			"org_public_id" => $orgPublicId,
		];
		
		if ($type != "custom") {
			$pushPayload["content_type"] = $notification["content_type"];
			$pushPayload["content_id"] = $notification["content_id"];
		}
		
		// If the notification is going to a user group / individuals find the device tokens
		// and add them to the push client options so that we target the correct users.
		
		if ($to === "user_group" or $to === "individuals") {
			
			$users = \Vinteract\Resources\User::with(["devices"])->hasAppPushSubscriptions()->find($notification["individuals"]);
			
			$devices = \Vinteract\Utils::mergeArrayOfArrays($users->pluck("devices")->toArray());
			
			$pushClientOptions["tokens"] = join(",", array_column($devices, "device_id"));
			
		}
		
		// Send the push notification.
		
		$pushClient = new \Vinteract\Notifications\AppPush();
		
		if ($customApp) {
			
			$pushClientOptions["api_key"] = \Vinteract\Client::getAPIKey("push_api_key");
			$pushClientOptions["channel"] = "alert";
			$pushClientOptions["payload"] = $pushPayload;
			
			$pushResponse = $pushClient->send($pushClientOptions);
			
		}
		
		if ($churchCentralApp) {
			
			$pushPayload["alert"] = "{$orgName}: {$body}";
			
			$pushClientOptions["api_key"] = \Vinteract\Client::getChurchCentralPushAPIKey();
			$pushClientOptions["channel"] = $orgDbPrefix;
			$pushClientOptions["payload"] = $pushPayload;
			
			$pushResponse = $pushClient->send($pushClientOptions);
			
		}
		
		// Push notification response.
		
		return $response->withJson($pushResponse->getResponseObject());
		
	})->add($authMiddleware);
	
	$app->post("/send/web", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$notification = $post["notification"];
		
		$to = $notification["to"];
		
		$responses = [];
		
		$subscriptions = [];
		
		$webPushPayload = json_encode([
			"title" => Client::getDisplayName(),
			"message" => $notification["body"],
			"url" => Client::getWebsiteURL(),
		]);
		
		// Get users.
		
		if ($to === "user_group" or $to === "individuals") {
			$users = \Vinteract\Resources\User::hasWebPushSubscriptions()->get($notification["individuals"]);
		} else {
			$users = \Vinteract\Resources\User::hasWebPushSubscriptions()->get();
		}
		
		// Instantiate web push client.
		
		$webPushClient = new WebPush([
			"VAPID" => [
				"subject" => Client::getDisplayName(),
				"publicKey" => Client::getAPIKey("web_push_public_vapid_key"),
				"privateKey" => Client::getAPIKey("web_push_private_vapid_key"),
			],
		], [], 30, ["verify" => false]);
		
		// Create the subscriptions and send the notifications.
		foreach ($users as $user) {
			foreach($user->devices as $device) {
				$subscriptions[] = Subscription::create([
					"endpoint" => $device["device_id"],
					"publicKey" => $device["meta"]["public_key"],
					"authToken" => $device["meta"]["auth_token"],
					"contentEncoding" => "aesgcm",
				]);
			}
		}
		
		// Send the notifications.
		
		foreach($subscriptions as $subscription) {
			$webPushClient->sendNotification($subscription, $webPushPayload, true);
		}
		
		// Flush the notifications and store the response.
		
		foreach ($webPushClient->flush() as $report) {
			if ($report->isSuccess()) {
				$responses[] = [ "success" => true, "error" => false, "message" => "Success" ];
			} else {
				$responses[] = [ "success" => false, "error" => true, "message" => $report->getReason() ];
			}
		}
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response(["code" => 200, "results" => $responses]))->getResponseObject());
		
	})->add($authMiddleware);
	
	$app->post("/send/sms", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$notification = $post["notification"];
		
		$userIds = $notification["individuals"];
		
		$users = \Vinteract\Resources\User::hasMobileNumber()->find($userIds);
		
		$responses = [];
		
		// Send the SMS notification(s).
		
		$smsClient = new Vinteract\Notifications\SMS();
		
		foreach ($users as $user) {
			
			$smsResponse = $smsClient->send([
				"mobile_number" => $user["phone"],
				"message" => $notification["body"] . " " . SMS_OPT_OUT_MESSAGE,
			]);
			
			$smsResponse->setContents([ "user" => $user ]);
			
			$responses[] = $smsResponse->getResponseObject();
			
		}
		
		// Update their SMS credits.
		
		Client::local("settings")->whereKey("sms_credits")->decrement("value", $notification["sms"]["credits_needed"]);
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"results" => $responses,
			"contents" => [
				"sms_credits" => Client::getSetting("sms_credits"),
			],
		]))->getResponseObject());
		
	})->add($authMiddleware);
	
	/*
		Content
	*/
	
	$contentCallback = function(Request $request, Response $response, array $args) {
		
		try {
			
			$type = $args["type"];
			
			if ($type === "prayer_requests") {
				$results = Client::local($type)
					->whereIn("status", ["Public", "Private"])
					->select(["id", "title", "body", "status"]);
			} else if ($type === "fan_wall") {
				$results = Client::local($type)
					->whereIn("status", ["Public", "Private"])
					->select(["id", "title", "body", "status"]);
			} else {
				$results = Client::local($type)
					->whereStatus("Published")
					->select(["id", "title", "body"]);
			}
			
			if (array_key_exists("id", $args)) {
				$id = $args["id"];
				if ($id === "latest") {
					$results->latest("added")->take(1);
				} elseif ($id === "soonest" and $type === "events") {
					$results->oldest("soonest_date")->take(1);
				} else {
					$results->whereId($id);
				}
			}
			
			$results = $results->get();
			
			return $response->withJson((new Vinteract\API\Response([
				"code" => count($results) ? 200 : 404,
				"results" => count($results) ? $results : null,
			]))->getResponseObject());
			
		} catch (\Exception $e) {
			
			return $response->withJson((new Vinteract\API\Response())->getResponseObject());
			
		}
		
	};
	
	$app->get("/content/{type}", $contentCallback)->add($authMiddleware);
	
	$app->get("/content/{type}/{id}", $contentCallback)->add($authMiddleware);
	
	/*
		User Groups
	*/
	
	$userGroupsCallback = function(Request $request, Response $response, array $args) {
		
		$results = \Vinteract\Resources\UserGroup::orderBy("title");
		
		if (array_key_exists("id", $args)) {
			$results->whereId($args["id"]);
		}
		
		if ($request->getQueryParam("type")) {
			$type = $request->getQueryParam("type");
			if ($type === "public") {
				$results->isPublic();
			} else {
				$results->isPrivate();
			}
		}
		
		$results = $results->get();
		
		return $response->withJson((new Vinteract\API\Response([
			"code" => count($results) ? 200 : 404,
			"results" => count($results) ? $results : null,
		]))->getResponseObject());
		
	};
	
	$app->get("/user-groups", $userGroupsCallback)->add($authMiddleware);
	
	$app->get("/user-groups/{id}", $userGroupsCallback)->add($authMiddleware);
	
	/*
		SMS Optout
	*/
	
	$app->post("/sms-opt-out", function(Request $request, Response $response, array $args) {
		
		// Variables
		
		$db = new \Vinteract\DB;
		
		$global = $db->capsule->getConnection("default");
		
		$post = $request->getParsedBody();
		
		// Parse mobile number.
		
		$phoneUtil = PhoneNumberUtil::getInstance();
		
		$mobileNumber = $phoneUtil->parse($post["mobile_number"], "AU");
		$mobileNumber = str_replace(" ", "", $phoneUtil->format($mobileNumber, PhoneNumberFormat::NATIONAL));
		
		// Search churches and unsubscribe the user's account from SMS notifications.
		
		$churches = $global->table("global_churches")->get();
		
		foreach ($churches as $church) {
			\Vinteract\Client::init($church->public_id);
			$user = \Vinteract\Resources\User::wherePhone($mobileNumber)->whereSmsNotifications(1)->first();
			if (isset($user)) {
				$user->sms_notifications = 0;
				$user->save();
			}
		}
		
		// Search businesses and unsubscribe the user's account from SMS notifications.
		
		$businesses = $global->table("global_businesses")->get();
		
		foreach ($businesses as $business) {
			\Vinteract\Client::init($business->public_id);
			$user = \Vinteract\Resources\User::wherePhone($mobileNumber)->whereSmsNotifications(1)->first();
			if (isset($user)) {
				$user->sms_notifications = 0;
				$user->save();
			}
		}
		
	});
	
	/*
		
		Catch-all route to serve a 404 Not Found page if none of the routes match
		NOTE: make sure this route is defined last
		
	*/
	
	$app->map(['GET', 'POST', 'PUT', 'DELETE', 'PATCH'], '/{routes:.+}', function($req, $res) {
	    $handler = $this->notFoundHandler; // handle using the default Slim page not found handler
	    return $handler($req, $res);
	});
	
	/*
		Run App
	*/
	
	$app->run();
	
