<?php
	
	require("vendor/autoload.php");
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Client;
	use \Vinteract\Utils;
	
	use \Vinteract\API\Response as ApiResponse;
	
	use \Vinteract\Auth\JwtToken;
	
	use \Vinteract\Resources\User;
	use \Vinteract\Resources\UserGroup;
	use \Vinteract\Resources\UserGroupUser;
	
	use \Vinteract\Resources\FollowersUserGroup;
	
	use \ReallySimpleJWT\Token;
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Environment
	*/
	
	$whitelist = array(
		"127.0.0.1",
		"::1"
	);
	
	if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
		define("ENVIRONMENT", "development");
	} else {
		define("ENVIRONMENT", "production");
	}
	
	/*
		Middleware
	*/
	
	$headerMiddleware = function($req, $res, $next) {
		
	    $response = $next($req, $res);
		
	    return $response
			->withHeader("Access-Control-Allow-Origin", "*")
			->withHeader("Access-Control-Allow-Headers", "Authorization, X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id")
			->withHeader("Access-Control-Allow-Methods", "GET, POST, PUT, DELETE, PATCH, OPTIONS")
			->withHeader("Access-Control-Expose-Headers", "Authorization");
		
	};
	
	$clientAuthMiddleware = function($request, $response, $next) {
		
		// Init client?
		
		$headers = $request->getHeader("Vinteract-Client-Id");
		
		if (count($headers)) {
			Client::init($headers[0]);
		}
		
		if (ENVIRONMENT === "development") {
			Client::init("EjNic");
		}
		
		// Return response.
		
		return $next($request, $response);
		
	};
	
	$userAuthMiddleware = function($request, $response, $next) {
		
		if ($request->getHeader("Authorization")) {
			
			$token = Utils::extractBearerToken($request->getHeader("Authorization")[0]);
			$secret = Client::getJWTSecretKey();
			
			$request = $request->withAttribute("user", User::whereUuid(JwtToken::getPayloadData($token, "data.user.uuid"))->first());
			
			if (!Token::validate($token, $secret)) {
				return $response->withJson((new Vinteract\API\Response([
					"code" => 401,
				]))->getResponseObject());
			}
			
		}
		
		return $next($request, $response);
		
	};
	
	$app->add($clientAuthMiddleware);
	$app->add($headerMiddleware);
	
	/*
		Followers
	*/
	
	$app->get("/followers", function(Request $request, Response $response, array $args) {
		
		$groups = FollowersUserGroup::all();
		
		if ($groups->isEmpty()) {
			return $response->withJson((new ApiResponse([
				"code" => 404,
			]))->getResponseObject());
		} else {
			return $response->withJson((new ApiResponse([
				"code" => 200,
				"results" => $groups,
			]))->getResponseObject());
		}
		
	});
	
	$app->post("/followers", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		// Lookup the user group and create it if it doesn't exist.
		// Otherwise return details for the existing user group.
		
		$userGroup = FollowersUserGroup::whereTitle($post["contentItem"]["title"])->get();
		
		if ($userGroup->isEmpty()) {
			
			$created = FollowersUserGroup::create([
				"title" => $post["contentItem"]["title"],
				"followers" => 1,
			]);
			
			if ($created) {
				return $response->withJson((new ApiResponse([
					"code" => 201,
					"contents" => [
						"user_group" => $created,
					],
				]))->getResponseObject());
			} else {
				return $response->withJson((new ApiResponse())->getResponseObject());
			}
			
		} else {
			
			return $response->withJson((new ApiResponse([
				"code" => 200,
				"contents" => [
					"user_group" => $userGroup->first(),
				],
			]))->getResponseObject());
			
		}
		
	})->add($userAuthMiddleware);
	
	/*
		Assign user
	*/
	
	$app->post("/assign/{uuid}", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$r = new ApiResponse();
		
		$userGroupUuid = $args["uuid"];
		
		$userId = null;
		
		// Validate the bearer token if it exists
		// and then extract the current user's id.
		
		if ($request->getHeader("Authorization")) {
			
			$token = Utils::extractBearerToken($request->getHeader("Authorization")[0]);
			$secret = Client::getJWTSecretKey();
			
			$tokenPayload = Token::getPayload($token, $secret);
			
			if (Token::validate($token, $secret)) {
				$userId = User::whereUuid($tokenPayload["data"]["user"]["uuid"])->first()->id;
			} else {
				return $response->withJson($r->getResponseObject());
			}
			
		}
		
		// Lookup the user group and then assign the user to it.
		
		$userGroup = UserGroup::whereUuid($userGroupUuid)->get();
		
		if (!$userGroup->isEmpty()) {
			$userGroupId = $userGroup->first()->id;
			$userGroupUser = UserGroupUser::whereUserId($userId)->whereUserGroupId($userGroupId)->get();
			if ($userGroupUser->isEmpty()) {
				$created = UserGroupUser::create(["user_id" => $userId, "user_group_id" => $userGroupId]);
				if ($created) {
					$r = new ApiResponse(["code" => 200]);
				}
			} else {
				$r = new ApiResponse(["code" => 200]);
			}
		}
		
		return $response->withJson($r->getResponseObject());
		
	});
	
	/*
		Unassign user
	*/
	
	$app->post("/unassign/{uuid}", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$user = $request->getAttribute("user");
		
		// Lookup the user group and if it exists
		// remove the user from it.
		
		$userGroup = UserGroup::whereUuid($args["uuid"])->get();
		
		if (!$userGroup->isEmpty()) {
			$deleted = UserGroupUser::whereUserId($user->id)->whereUserGroupId($userGroup->first()->id)->delete();
			if ($deleted) {
				return $response->withJson((new ApiResponse(["code" => 200]))->getResponseObject());
			}
		}
		
		return $response->withJson((new ApiResponse())->getResponseObject());
		
	})->add($userAuthMiddleware);
	
	/*
		Unassign user (2)
	*/
	
	$app->post("/unassign/{userUuid}/{userGroupUuid}", function(Request $request, Response $response, array $args) {
		
		// Get the user's ID.
		
		$userId = User::whereUuid($args["userUuid"])->first()->id;
		
		// Lookup the user group and if it exists
		// remove the user from it.
		
		$userGroup = UserGroup::whereUuid($args["userGroupUuid"])->get();
		
		if (isset($userId) and !$userGroup->isEmpty()) {
			$deleted = UserGroupUser::whereUserId($userId)->whereUserGroupId($userGroup->first()->id)->delete();
			if ($deleted) {
				return $response->withJson((new ApiResponse(["code" => 200]))->getResponseObject());
			}
		}
		
		return $response->withJson((new ApiResponse())->getResponseObject());
		
	});
	
	/*
		
		Catch-all route to serve a 404 Not Found page if none of the routes match
		NOTE: make sure this route is defined last
		
	*/
	
	$app->map(['GET', 'POST', 'PUT', 'DELETE', 'PATCH'], '/{routes:.+}', function($req, $res) {
	    $handler = $this->notFoundHandler; // handle using the default Slim page not found handler
	    return $handler($req, $res);
	});
	
	/*
		Run App
	*/
	
	$app->run();
	