<?php
	
	use \Psr\Http\Message\ServerRequestInterface as Request;
	use \Psr\Http\Message\ResponseInterface as Response;
	
	use \Vinteract\Client as Client;
	
	require("vendor/autoload.php");
	
	$config = new \Slim\Container([
		"settings" => [
			"displayErrorDetails" => true,
		],
	]);
	
	$app = new \Slim\App($config);
	
	/*
		Environment
	*/
	
	$whitelist = array(
		"127.0.0.1",
		"::1"
	);

	if(in_array($_SERVER["REMOTE_ADDR"], $whitelist)){
		define("ENVIRONMENT", "development");
	} else {
		define("ENVIRONMENT", "production");
	}
	
	/*
		Middleware
	*/
	
	$app->add(function ($req, $res, $next) {
	    $response = $next($req, $res);
	    return $response
			->withHeader('Access-Control-Allow-Origin', '*')
			->withHeader('Access-Control-Allow-Headers', 'X-Requested-With, Content-Type, Accept, Origin, Vinteract-Client-Id')
			->withHeader('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, PATCH, OPTIONS');
	});
	
	$app->add(function($request, $response, $next) {
		
		// Init client?
		
		$headers = $request->getHeader("Vinteract-Client-Id");
		
		if (count($headers)) {
			Vinteract\Client::init($headers[0]);
		} else {
			Vinteract\Client::init("EjNic");
		}
		
		// Return response.
		
		return $next($request, $response);
		
	});

	/*
		Functions
	*/

	$slugify = function($text, string $divider = '-') {

		// replace non letter or digits by divider
		$text = preg_replace('~[^\pL\d]+~u', $divider, $text);

		// transliterate
		$text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);

		// remove unwanted characters
		$text = preg_replace('~[^-\w]+~', '', $text);

		// trim
		$text = trim($text, $divider);

		// remove duplicate divider
		$text = preg_replace('~-+~', $divider, $text);

		// lowercase
		$text = strtolower($text);

		if (empty($text)) {
			return 'n-a';
		}

		return $text;
	};

	$updateRoutes = function($routes, $content_type = "", $label = "") use($slugify) {

		/*
			Variables.
		*/
		
		$route = $slugify($label);
		
		/*
			Delete old routes.
		*/
		
		if (array_key_exists($content_type, $routes["content_type_to_route"])) {
			
			// Get old route name.
			
			$old_route = $routes["content_type_to_route"][$content_type];
			
			// Remove content type to route.
			
			unset($routes["content_type_to_route"][$content_type]);
			
			// Remove route to content type?
			
			if ($content_type == "prayer_requests" or $content_type == "fan_wall") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				unset($routes["route_to_content_type"]["{$old_route}/submit"]);
				
			} elseif ($content_type == "events") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				unset($routes["route_to_content_type"]["{$old_route}/rsvp"]);
				unset($routes["route_to_content_type"]["{$old_route}/rsvp/(:any)"]);
				
			} elseif ($content_type == "enquiries") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/post"]);
				unset($routes["route_to_content_type"]["{$old_route}/topic"]);
				
			} elseif ($content_type == "payments") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} elseif ($content_type == "subsriptions") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} elseif ($content_type == "chat_rooms") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} elseif ($content_type == "gallery") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} elseif ($content_type == "collections") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} elseif ($content_type == "asa_programs") {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			} else {
				
				unset($routes["route_to_content_type"]["{$old_route}"]);
				unset($routes["route_to_content_type"]["{$old_route}/(:any)"]);
				
			}
			
		}
		
		/*
			Create new routes.
		*/
		
		$routes["content_type_to_route"][$content_type] = $route;
		
		if ($content_type == "prayer_requests" or $content_type == "fan_wall") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "content_list");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "content_list/index/$1");
			$routes["route_to_content_type"]["{$route}/submit"] = array("content_type" => $content_type, "route" => "{$content_type}_submit/index/$1");
			
		} elseif ($content_type == "events") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "content_list");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "content_list/index/$1");
			$routes["route_to_content_type"]["{$route}/rsvp"] = array("content_type" => $content_type, "route" => "events/rsvp/$1");
			$routes["route_to_content_type"]["{$route}/rsvp/(:any)"] = array("content_type" => $content_type, "route" => "events/rsvp/$1");
			
		} elseif ($content_type == "enquiries") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "enquiries/index/$1");
			$routes["route_to_content_type"]["{$route}/post"] = array("content_type" => $content_type, "route" => "enquiries/post/$1");
			$routes["route_to_content_type"]["{$route}/topic"] = array("content_type" => $content_type, "route" => "enquiries/topic/$1");
			
		} elseif ($content_type == "payments") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "payments");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "payments/index/$1");
			
		} elseif ($content_type == "subscriptions") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "subscriptions");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "subscriptions/index/$1");
			
		} elseif ($content_type == "chat_rooms") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "chat/index/$1");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "chat/conversation/$1");
			
		} elseif ($content_type == "gallery") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "gallery/index/$1");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "gallery/album/$1");
			
		} elseif ($content_type == "collections") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "collections");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "collections/collection/$1");
			
		} elseif ($content_type == "asa_programs") {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "content_list_programs");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "content_list_programs/index/$1");
			
		} else {
			
			$routes["route_to_content_type"]["{$route}"] = array("content_type" => $content_type, "route" => "content_list");
			$routes["route_to_content_type"]["{$route}/(:any)"] = array("content_type" => $content_type, "route" => "content_list/index/$1");
			
		}

		return $routes;
		
	};
	
	/*
		Base
	*/
	
	$app->post("/", function(Request $request, Response $response, array $args) use($updateRoutes) {
		
		// Variables.
		
		$post = $request->getParsedBody();
		
		$settings = $post["settings"];
		
		// Update content features.
		
		if (array_key_exists("content", $settings)) {
			foreach($settings["content"] as $feature) {
				if (array_key_exists("permission_key", $feature)) {
					Client::local("settings_features")->updateOrInsert(["key" => $feature["permission_key"]], ["value" => $feature["permission"]]);
				}
				if (array_key_exists("label_key", $feature)) {

					// LEG-279

					Client::local("settings_labels")->updateOrInsert(
						[
							"key" => $feature["label_key"]
						], 
						[
							"value" => $feature["label"], 
						]
					);

					if (isset($feature["show_date_added"])) {
						Client::local("settings")->updateOrInsert(
							[
								"key" => $feature["label_key"] . "_show_date_added"
							], 
							[
								"value" => $feature["show_date_added"], 
							]
						);
					}

				}
			}
		}

		// Update content visibility LEG-259

		if (array_key_exists("content_visibility", $settings)) {
			foreach($settings["content_visibility"] as $index=>$content_visibility) {
				
				Client::local("settings_contents_visibility")->updateOrInsert(
					[
						"content_type" => $index
					], 
					[
						"content_visibility" => intval($content_visibility["content_visibility"]), 
						"checked_private_user_group" => json_encode($content_visibility["checked_private_user_group"]), 
						"checked_public_user_group" => json_encode($content_visibility["checked_public_user_group"])
					]
				);
				
			}
		}
		
		// Update informational features.
		
		if (array_key_exists("information", $settings)) {
			foreach($settings["information"] as $feature) {
				if (array_key_exists("permission_key", $feature)) {
					Client::local("settings_features")->where("key", $feature["permission_key"])->update(["value" => $feature["permission"]]);
				}
				if (array_key_exists("label_key", $feature)) {

					// bug fix VS-674

					if ($feature["label_key"] == 'locations') {
						Client::local("settings_labels")->where("key", 'locations1')->update(["value" => $feature["label"]]);
					} else {
						Client::local("settings_labels")->where("key", $feature["label_key"])->update(["value" => $feature["label"]]);
					}
				}
			}
		}
		
		// Update kids corner.
		
		if (array_key_exists("kids", $settings)) {
			foreach($settings["kids"] as $feature) {
				if (array_key_exists("permission_key", $feature)) {
					Client::local("settings_features")->where("key", $feature["permission_key"])->update(["value" => $feature["permission"]]);
				}
				if (array_key_exists("label_key", $feature)) {
					Client::local("settings_labels")->where("key", $feature["label_key"])->update(["value" => $feature["label"]]);
				}
			}
		}
		
		// Update premium features.
		
		if (array_key_exists("premium", $settings)) {
			foreach($settings["premium"] as $feature) {
				if (array_key_exists("permission_key", $feature)) {
					Client::local("settings_features")->where("key", $feature["permission_key"])->update(["value" => $feature["permission"]]);
				}
				if (array_key_exists("label_key", $feature)) {
					Client::local("settings_labels")->where("key", $feature["label_key"])->update(["value" => $feature["label"]]);
				}
			}
		}

		// Update content filters.
		
		if (array_key_exists("filters", $settings)) {
			foreach($settings["filters"] as $feature) {
				if (array_key_exists("permission_key", $feature)) {
					Client::local("settings_features")->updateOrInsert([
						"key" => $feature["permission_key"]
					],[
						"value" => $feature["permission"]
					]);
				}
			}
		}
		
		// Update events.
		
		if (array_key_exists("events", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["events"]["permission_key"])->update(["value" => $settings["events"]["permission"]]);
			Client::local("settings_features")->where("key", $settings["events"]["rsvp_permission_key"])->update(["value" => $settings["events"]["rsvp_permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["events"]["label_key"])->update(["value" => $settings["events"]["label"]]);
			
			Client::local("settings_events")->update(["recipient_address" => implode(",", $settings["events"]["rsvp_recipients"])]);
			
		}
		
		// Update prayer requests.
		
		if (array_key_exists("prayer_requests", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["prayer_requests"]["permission_key"])->update(["value" => $settings["prayer_requests"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["prayer_requests"]["label_key"])->update(["value" => $settings["prayer_requests"]["label"]]);
			Client::local("settings_labels")->where("key", "compose_prayer_request")->update(["value" => $settings["prayer_requests"]["compose_prayer_request"]]);
			Client::local("settings_labels")->where("key", "private_prayer_request_confirmation")->update(["value" => $settings["prayer_requests"]["private_prayer_request_confirmation"]]);
			Client::local("settings_labels")->where("key", "public_prayer_request_confirmation")->update(["value" => $settings["prayer_requests"]["public_prayer_request_confirmation"]]);
			
			Client::local("settings_prayer_requests")->update(["recipient_address" => implode(",", $settings["prayer_requests"]["recipients"])]);

			if ($settings["prayer_requests"]["automatic_pastoral_care_action"]) {
				Client::local("settings")->updateOrInsert(["key" => "automatic_pastoral_care_action"], ["value" => $settings["prayer_requests"]["automatic_pastoral_care_action"]]);
			}
			
		}
		
		// Update fan wall.
		
		if (array_key_exists("fan_wall", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["fan_wall"]["permission_key"])->update(["value" => $settings["fan_wall"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["fan_wall"]["label_key"])->update(["value" => $settings["fan_wall"]["label"]]);
			Client::local("settings_labels")->where("key", "compose_fan_post")->update(["value" => $settings["fan_wall"]["compose_fan_post"]]);
			
			Client::local("settings_fan_wall")->update(["recipient_address" => implode(",", $settings["fan_wall"]["recipients"])]);
			
		}
		
		// Update devotionals.
		
		if (array_key_exists("devotionals", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["devotionals"]["permission_key"])->update(["value" => $settings["devotionals"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["devotionals"]["label_key"])->update(["value" => $settings["devotionals"]["label"]]);
			
			Client::local("settings_devotionals")->update([
				"show_date_added" => $settings["devotionals"]["show_date_added"],
				"source" => $settings["devotionals"]["source"],
				"url" => $settings["devotionals"]["url"]
			]);
			
		}
		
		// Update giving.
		
		if (array_key_exists("giving", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["giving"]["permission_key"])->update(["value" => $settings["giving"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["giving"]["label_key"])->update(["value" => $settings["giving"]["label"]]);
			
			Client::local("settings_giving")->update(["url" => $settings["giving"]["url"]]);
			
		}
		
		// Update blog.
		
		if (array_key_exists("blog", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["blog"]["permission_key"])->update(["value" => $settings["blog"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["blog"]["label_key"])->update(["value" => $settings["blog"]["label"]]);
			
			Client::local("settings_blog")->update(["url" => $settings["blog"]["url"]]);
			
		}
		
		// Update shop.
		
		if (array_key_exists("shop", $settings)) {
			
			Client::local("settings_features")->where("key", $settings["shop"]["permission_key"])->update(["value" => $settings["shop"]["permission"]]);
			
			Client::local("settings_labels")->where("key", $settings["shop"]["label_key"])->update(["value" => $settings["shop"]["label"]]);
			
			Client::local("settings_shop")->update(["url" => $settings["shop"]["url"]]);
			
		}
		
		// Update contact details.
		
		if (array_key_exists("contact", $settings)) {
			foreach($settings["contact"] as $key => $value) {
				Client::local("settings_home_other")->updateOrInsert([
					"key" => $key
				],[
					"value" => $value
				]);
			}
		}

		// Update sms settings.
		
		if (array_key_exists("sms_settings", $settings) && array_key_exists("sms_credit_reminder_emails", $settings["sms_settings"])) {
			foreach($settings["sms_settings"] as $key => $value) {
			
				Client::local("settings")->updateOrInsert([
					"key" => "sms_credit_reminder_emails"
				],[
					"value" =>  json_encode($settings["sms_settings"]["sms_credit_reminder_emails"])
				]);
			
			}
		}

		if (array_key_exists("sms_settings", $settings) && array_key_exists("reminder_balance_level", $settings["sms_settings"])) {
			foreach($settings["sms_settings"] as $key => $value) {
			
				Client::local("settings")->updateOrInsert([
					"key" => "reminder_balance_level"
				],[
					"value" =>  $settings["sms_settings"]["reminder_balance_level"]
				]);
			
			}
		}
		
		// Update social media.
		
		if (array_key_exists("social_media", $settings)) {
			foreach($settings["social_media"] as $key => $value) {
				Client::local("settings_social_media")->updateOrInsert([
					"key" => $key
				],[
					"value" => $value
				]);
			}
		}

		// Variables.
			
		$domain = Client::getData("website_path");
		$routes_path = "../../{$domain}/config/routes.json";
					
		// Load routes file or create an empty routes array.
		
		if (file_exists($routes_path)) {
			$routes = json_decode(file_get_contents($routes_path), true);
		} else {
			$routes = array(
				"content_type_to_route" => array(),
				"route_to_content_type" => array(),
			);
		}

		if (Client::getType() === "churches") {	

			// Update routes.
			
			$routes = $updateRoutes($routes, "verses", $settings["content"]["verses"]["label"]);
			
			$routes = $updateRoutes($routes, "devotionals", $settings["devotionals"]["label"]);
			
			$routes = $updateRoutes($routes, "news", $settings["content"]["news"]["label"]);
			
			$routes = $updateRoutes($routes, "newsletters", $settings["content"]["newsletters"]["label"]);
			
			$routes = $updateRoutes($routes, "sermons", $settings["content"]["sermons"]["label"]);
			
			$routes = $updateRoutes($routes, "events", $settings["events"]["label"]);
			
			$routes = $updateRoutes($routes, "prayer_requests", $settings["prayer_requests"]["label"]);
			
			$routes = $updateRoutes($routes, "gallery", $settings["content"]["gallery"]["label"]);
			
			$routes = $updateRoutes($routes, "about", $settings["information"]["about"]["label"]);
			
			$routes = $updateRoutes($routes, "ministries", $settings["information"]["ministries"]["label"]);
			
			$routes = $updateRoutes($routes, "team", $settings["information"]["team"]["label"]);
			
			$routes = $updateRoutes($routes, "church_locations", $settings["information"]["church_locations"]["label"]);
			
			$routes = $updateRoutes($routes, "enquiries", $settings["information"]["enquiries"]["label"]);
									
			// Fixed: 30-05-2022 Add three new features under Standard and Visible in Menu.
			$routes = $updateRoutes($routes, "custom_section_1", $settings["content"]["custom_section_1"]["label"]);
			
			$routes = $updateRoutes($routes, "custom_section_2", $settings["content"]["custom_section_2"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_3", $settings["content"]["custom_section_3"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_4", $settings["content"]["custom_section_4"]["label"]);
			
			$routes = $updateRoutes($routes, "custom_section_5", $settings["content"]["custom_section_5"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_6", $settings["content"]["custom_section_6"]["label"]);
						
		} else if (Client::getType() === "businesses") {

			// Update routes.
			
			$routes = $updateRoutes($routes, "advice", $settings["content"]["advice"]["label"]);
			
			$routes = $updateRoutes($routes, "news", $settings["content"]["news"]["label"]);

			$routes = $updateRoutes($routes, "asa_programs", $settings["content"]["programs"]["label"]);
			
			$routes = $updateRoutes($routes, "newsletters", $settings["content"]["newsletters"]["label"]);
			
			$routes = $updateRoutes($routes, "promotional_content", $settings["content"]["promotional_content"]["label"]);
			
			$routes = $updateRoutes($routes, "events", $settings["events"]["label"]);
			
			$routes = $updateRoutes($routes, "fan_wall", $settings["fan_wall"]["label"]);
			
			$routes = $updateRoutes($routes, "offers", $settings["content"]["offers"]["label"]);

			$routes = $updateRoutes($routes, "gallery", $settings["content"]["gallery"]["label"]);
			
			$routes = $updateRoutes($routes, "about", $settings["information"]["about"]["label"]);
			
			$routes = $updateRoutes($routes, "services", $settings["information"]["services"]["label"]);
			
			$routes = $updateRoutes($routes, "team", $settings["information"]["team"]["label"]);
			
			$routes = $updateRoutes($routes, "locations", $settings["information"]["locations"]["label"]);
			
			$routes = $updateRoutes($routes, "enquiries", $settings["information"]["enquiries"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_1", $settings["content"]["custom_section_1"]["label"]);
			
			$routes = $updateRoutes($routes, "custom_section_2", $settings["content"]["custom_section_2"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_3", $settings["content"]["custom_section_3"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_4", $settings["content"]["custom_section_4"]["label"]);
			
			$routes = $updateRoutes($routes, "custom_section_5", $settings["content"]["custom_section_5"]["label"]);

			$routes = $updateRoutes($routes, "custom_section_6", $settings["content"]["custom_section_6"]["label"]);
								
		}

		// Save routes.
			
		if (file_exists($routes_path)) {
			file_put_contents($routes_path, json_encode($routes));
		}
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [
				"settings" => $request->getParsedBody(),
			],
		]))->getResponseObject());
		
	});
	
	/*
		Init
	*/
	
	$app->get("/init", function(Request $request, Response $response, array $args) {
		
		// Variables.
		
		$settings = [];
		$diay_only = false;

		// Diary Only

		if (Client::getType() === "businesses") {
			$client = Client::global("global_" . Client::getType())->find(Client::getId());

			$diay_only = $settings["diary_only"] = $client->diary_only == "1" ? true : false;
		}
		
		// General settings.

		// VIF-124
		//------------------start-----------------------
		$enble_website = Client::getData("website");
		$enble_church_central = Client::getData("church_central");
		$enble_custom_app = Client::getData("custom_app");

		$settings["enable_features_section"] = $enble_website == 1 || $enble_church_central == 1 || $enble_custom_app == 1;
		//-----------------end------------------------

		$permissions = array_column(Client::local("settings_features")->get()->toArray(), "value", "key");
		
		$labels = array_column(Client::local("settings_labels")->get()->toArray(), "value", "key");

		$sms_settings = array_column(Client::local("settings")->get()->toArray(), "value", "key");

		$settings["permissions"] = $permissions;
		
		$settings["labels"] = $labels;

		if (array_key_exists("sms_credit_reminder_emails", $sms_settings)) {
			$sms_settings["sms_credit_reminder_emails"] = json_decode($sms_settings["sms_credit_reminder_emails"], 1);
		}
		
		$settings["sms_settings"] = $sms_settings;

		$settings["social_media"] = array_column(Client::local("settings_social_media")->get()->toArray(), "value", "key");
		
		$settings["contact"] = array_column(Client::local("settings_home_other")->get()->toArray(), "value", "key");
		
		// Content Visibility LEG-259
		// User Group

		$settings["user_groups"] = Client::local("user_groups")->get()->toArray();
		
		$settings_contents_visibility = Client::local("settings_contents_visibility")->get()->toArray();
		
		// if ( !empty($settings_contents_visibility) ) {
			foreach ($settings_contents_visibility as $index=>$scv) {

				if ( !empty($scv->checked_private_user_group) ) {
					$settings_contents_visibility[$index]->checked_private_user_group = json_decode($scv->checked_private_user_group, 1);
				}
	
				if ( !empty($scv->checked_public_user_group) ) {
					$settings_contents_visibility[$index]->checked_public_user_group = json_decode($scv->checked_public_user_group, 1);
				}
			}
			
			$settings["content_visibility"] = array_combine(array_column($settings_contents_visibility, 'content_type'), $settings_contents_visibility);

			if (!isset($settings["content_visibility"]["news"])) {
				$settings["content_visibility"]["news"] = [
					"content_type" => "news",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_1"])) {
				$settings["content_visibility"]["custom_section_1"] = [
					"content_type" => "custom_section_1",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_2"])) {
				$settings["content_visibility"]["custom_section_2"] = [
					"content_type" => "custom_section_2",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_3"])) {
				$settings["content_visibility"]["custom_section_3"] = [
					"content_type" => "custom_section_3",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_4"])) {
				$settings["content_visibility"]["custom_section_4"] = [
					"content_type" => "custom_section_4",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_5"])) {
				$settings["content_visibility"]["custom_section_5"] = [
					"content_type" => "custom_section_5",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}

			if (!isset($settings["content_visibility"]["custom_section_6"])) {
				$settings["content_visibility"]["custom_section_6"] = [
					"content_type" => "custom_section_6",
					"checked_private_user_group" => null,
					"checked_public_user_group" => null,
					"content_visibility" => 0,
				];
			}
		// } else {
		// 	$settings["content_visibility"]["news"] = [
		// 		"content_type" => "news",
		// 		"checked_private_user_group" => null,
		// 		"checked_public_user_group" => null,
		// 		"content_visibility" => 0,
		// 	];
		// }
		
		// General features.
		
		$settings["content"]["home"] = [
			"permission" => $permissions["content_home"],
			"permission_key" => "content_home",
			"permission_editable" => false,
			"label" => $labels["home"],
			"label_original" => "Home Screen",
			"label_key" => "home",
			"label_editable" => true,
		];

		$settings["content"]["news"] = [
			"permission" => $permissions["content_news"],
			"permission_key" => "content_news",
			"permission_editable" => $diay_only ? false : true,
			"label" => $labels["news"],
			"label_original" => "News",
			"label_key" => "news",
			"label_editable" => true,
		];
		
		$settings["content"]["newsletters"] = [
			"permission" => $permissions["content_newsletters"],
			"permission_key" => "content_newsletters",
			"permission_editable" => $diay_only ? false : true,
			"label" => $labels["newsletters"],
			"label_original" => "Newsletters",
			"label_key" => "newsletters",
			"label_editable" => true,
		];
		
		$settings["content"]["gallery"] = [
			"permission" => $permissions["content_gallery"],
			"permission_key" => "content_gallery",
			"permission_editable" => $diay_only ? false : true,
			"label" => $labels["gallery"],
			"label_original" => "Photo Gallery",
			"label_key" => "gallery",
			"label_editable" => true,
		];
		
		// Church features.

		if (Client::getType() === "churches") {
			
			// Standard features.
			
			$settings["content"]["verses"] = [
				"permission" => $permissions["content_verses"],
				"permission_key" => "content_verses",
				"permission_editable" => true,
				"label" => $labels["verses"],
				"label_original" => "Daily Verse",
				"label_key" => "verses",
				"label_editable" => true,
			];
			
			$settings["content"]["sermons"] = [
				"permission" => $permissions["content_sermons"],
				"permission_key" => "content_sermons",
				"permission_editable" => true,
				"label" => $labels["sermons"],
				"label_original" => "Sermons",
				"label_key" => "sermons",
				"label_editable" => true,
			];

			// Custom Section 1,2,3 LEG-279

			$settings["content"]["custom_section_1"] = [
				"permission" => $permissions["content_custom_section_1"],
				"permission_key" => "content_custom_section_1",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_1"],
				"label_original" => "Custom Section 1",
				"label_key" => "custom_section_1",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_1_show_date_added"]
			];

			$settings["content"]["custom_section_2"] = [
				"permission" => $permissions["content_custom_section_2"],
				"permission_key" => "content_custom_section_2",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_2"],
				"label_original" => "Custom Section 2",
				"label_key" => "custom_section_2",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_2_show_date_added"]
			];

			$settings["content"]["custom_section_3"] = [
				"permission" => $permissions["content_custom_section_3"],
				"permission_key" => "content_custom_section_3",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_3"],
				"label_original" => "Custom Section 3",
				"label_key" => "custom_section_3",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_3_show_date_added"]
			];

			$settings["content"]["custom_section_4"] = [
				"permission" => $permissions["content_custom_section_4"],
				"permission_key" => "content_custom_section_4",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_4"],
				"label_original" => "Custom Section 4",
				"label_key" => "custom_section_4",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_4_show_date_added"]
			];

			$settings["content"]["custom_section_5"] = [
				"permission" => $permissions["content_custom_section_5"],
				"permission_key" => "content_custom_section_5",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_5"],
				"label_original" => "Custom Section 5",
				"label_key" => "custom_section_5",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_5_show_date_added"]
			];

			$settings["content"]["custom_section_6"] = [
				"permission" => $permissions["content_custom_section_6"],
				"permission_key" => "content_custom_section_6",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_6"],
				"label_original" => "Custom Section 6",
				"label_key" => "custom_section_6",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_6_show_date_added"]
			];
			
			$settings["kids"]["kids"] = [
				"permission" => $permissions["content_kids"],
				"permission_key" => "content_kids",
				"permission_editable" => true,
				"label" => $labels["kids"],
				"label_original" => "Kids Corner",
				"label_key" => "kids",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_news"] = [
				"permission" => $permissions["content_kids_news"],
				"permission_key" => "content_kids_news",
				"permission_editable" => true,
				"label" => $labels["kids_news"],
				"label_original" => "Kids News",
				"label_key" => "kids_news",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_verses"] = [
				"permission" => $permissions["content_kids_verses"],
				"permission_key" => "content_kids_verses",
				"permission_editable" => true,
				"label" => $labels["kids_verses"],
				"label_original" => "Kids Verses",
				"label_key" => "kids_verses",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_sermons"] = [
				"permission" => $permissions["content_kids_sermons"],
				"permission_key" => "content_kids_sermons",
				"permission_editable" => true,
				"label" => $labels["kids_sermons"],
				"label_original" => "Kids Sermons",
				"label_key" => "kids_sermons",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_devotionals"] = [
				"permission" => $permissions["content_kids_devotionals"],
				"permission_key" => "content_kids_devotionals",
				"permission_editable" => true,
				"label" => $labels["kids_devotionals"],
				"label_original" => "Kids Devotionals",
				"label_key" => "kids_devotionals",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_events"] = [
				"permission" => $permissions["content_kids_events"],
				"permission_key" => "content_kids_events",
				"permission_editable" => true,
				"label" => $labels["kids_events"],
				"label_original" => "Kids Events",
				"label_key" => "kids_events",
				"label_editable" => true,
			];
			
			$settings["kids"]["kids_newsletters"] = [
				"permission" => $permissions["content_kids_newsletters"],
				"permission_key" => "content_kids_newsletters",
				"permission_editable" => true,
				"label" => $labels["kids_newsletters"],
				"label_original" => "Kids Newsletters",
				"label_key" => "kids_newsletters",
				"label_editable" => true,
			];
			
			$settings["information"]["about"] = [
				"permission" => $permissions["us_about"],
				"permission_key" => "us_about",
				"permission_editable" => true,
				"label" => $labels["about"],
				"label_original" => "About Us",
				"label_key" => "about",
				"label_editable" => true,
			];
			
			$settings["information"]["ministries"] = [
				"permission" => $permissions["us_ministries"],
				"permission_key" => "us_ministries",
				"permission_editable" => true,
				"label" => $labels["ministries"],
				"label_original" => "Ministries",
				"label_key" => "ministries",
				"label_editable" => true,
			];
			
			$settings["information"]["team"] = [
				"permission" => $permissions["us_team"],
				"permission_key" => "us_team",
				"permission_editable" => true,
				"label" => $labels["team"],
				"label_original" => "Team Member",
				"label_key" => "team",
				"label_editable" => true,
			];
			
			$settings["information"]["church_locations"] = [
				"permission" => $permissions["us_church_locations"],
				"permission_key" => "us_church_locations",
				"permission_editable" => true,
				"label" => $labels["locations1"],
				"label_original" => "Church Locations",
				// Fixed: LEG-192 07-12-2021 Some of the custom label does not reflect in the App and Website
				"label_key" => "locations1",
				"label_editable" => true,
			];
			
			$settings["information"]["enquiries"] = [
				"permission" => $permissions["us_enquiries"],
				"permission_key" => "us_enquiries",
				"permission_editable" => true,
				"label" => $labels["enquiries"],
				"label_original" => "Enquiries",
				"label_key" => "enquiries",
				"label_editable" => true,
			];
			
			// Events.
			
			$settings_events = Client::local("settings_events")->select("recipient_address")->first();
			
			if (strlen($settings_events->recipient_address)) {
				$rsvp_recipients = explode(",", $settings_events->recipient_address);
			} else {
				$rsvp_recipients = [];
			}
			
			$settings["events"] = [
				
				"permission" => $permissions["content_events"],
				"permission_key" => "content_events",
				"permission_editable" => true,
				
				"label" => $labels["events"],
				"label_original" => "Events",
				"label_key" => "events",
				"label_editable" => true,
				
				"rsvp_recipients" => $rsvp_recipients,
				"rsvp_permission" => $permissions["content_rsvp"],
				"rsvp_permission_key" => "content_rsvp",
				"rsvp_label" => "RSVP",
				
			];
			
			// Prayer requests.
			
			$settings_prayer_requests = Client::local("settings_prayer_requests")->select("recipient_address")->first();
			
			if (strlen($settings_prayer_requests->recipient_address)) {
				$prayer_request_recipients = explode(",", $settings_prayer_requests->recipient_address);
			} else {
				$prayer_request_recipients = [];
			}
			
			$settings["prayer_requests"] = [
				
				"permission" => $permissions["content_prayer_requests"],
				"permission_key" => "content_prayer_requests",
				"permission_editable" => true,
				"label" => $labels["prayer_requests"],
				"label_original" => "Prayer Requests",
				"label_key" => "prayer_requests",
				"label_editable" => true,
				
				"recipients" => $prayer_request_recipients,
				
				"compose_prayer_request" => $labels["compose_prayer_request"],
				"public_prayer_request_confirmation" => $labels["public_prayer_request_confirmation"],
				"private_prayer_request_confirmation" => $labels["private_prayer_request_confirmation"],

				"automatic_pastoral_care_action" => $sms_settings["automatic_pastoral_care_action"]
			];
			
			// Devotionals.
			
			$settings_devotionals = Client::local("settings_devotionals")->first();
			
			$settings["devotionals"] = [
				
				"permission" => $permissions["content_devotionals"],
				"permission_key" => "content_devotionals",
				"permission_editable" => true,
				"label" => $labels["devotionals"],
				"label_original" => "Devotionals",
				"label_key" => "devotionals",
				"label_editable" => true,
				
				"source" => $settings_devotionals->source,
				"url" => $settings_devotionals->url,
				"show_date_added" => $settings_devotionals->show_date_added,
				
			];
			
			// Giving.
			
			// $settings["giving"] = [
				
			// 	"permission" => $permissions["us_giving"],
			// 	"permission_key" => "us_giving",
			// 	"permission_editable" => true,
			// 	"label" => $labels["giving"],
			// 	"label_original" => "Third Party Payment", // CC22-253
			// 	"label_key" => "giving",
			// 	"label_editable" => true,
				
			// 	"url" => Client::local("settings_giving")->select("url")->value("url"),
				
			// ];
			
			// Blog.
			
			$settings["blog"] = [
				
				"permission" => $permissions["us_blog"],
				"permission_key" => "us_blog",
				"permission_editable" => true,
				
				"label" => $labels["blog"],
				"label_original" => "Blog",
				"label_key" => "blog",
				"label_editable" => true,
				
				"url" => Client::local("settings_blog")->select("url")->value("url"),
				
			];
			
		}
		
		// Business features.	
		
		if (Client::getType() === "businesses") {

			// Content features.
			
			$settings["content"]["advice"] = [
				"permission" => $permissions["content_advice"],
				"permission_key" => "content_advice",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["advice"],
				"label_original" => "Business Tips & Advice	",
				"label_key" => "advice",
				"label_editable" => true,
			];
			
			$settings["content"]["promotional_content"] = [
				"permission" => $permissions["content_promotional_content"],
				"permission_key" => "content_promotional_content",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["promotional_content"],
				"label_original" => "Promotional Content",
				"label_key" => "promotional_content",
				"label_editable" => true,
			];
			
			$settings["content"]["offers"] = [
				"permission" => $permissions["content_offers"],
				"permission_key" => "content_offers",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["offers"],
				"label_original" => "Offers",
				"label_key" => "offers",
				"label_editable" => true,
			];
			
			$settings["content"]["programs"] = [
				"permission" => $permissions["content_asa_programs"],
				"permission_key" => "content_asa_programs",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["asa_programs"],
				"label_original" => "Programs",
				"label_key" => "asa_programs",
				"label_editable" => true,
			];

			$settings["content"]["custom_section_1"] = [
				"permission" => $permissions["content_custom_section_1"],
				"permission_key" => "content_custom_section_1",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_1"],
				"label_original" => "Custom Section 1 (Web Only)",
				"label_key" => "custom_section_1",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_1_show_date_added"]
			];

			$settings["content"]["custom_section_2"] = [
				"permission" => $permissions["content_custom_section_2"],
				"permission_key" => "content_custom_section_2",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_2"],
				"label_original" => "Custom Section 2 (Web Only)",
				"label_key" => "custom_section_2",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_2_show_date_added"]
			];

			$settings["content"]["custom_section_3"] = [
				"permission" => $permissions["content_custom_section_3"],
				"permission_key" => "content_custom_section_3",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_3"],
				"label_original" => "Custom Section 3 (Web Only)",
				"label_key" => "custom_section_3",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_3_show_date_added"]
			];

			$settings["content"]["custom_section_4"] = [
				"permission" => $permissions["content_custom_section_4"],
				"permission_key" => "content_custom_section_4",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_4"],
				"label_original" => "Custom Section 4",
				"label_key" => "custom_section_4",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_4_show_date_added"]
			];

			$settings["content"]["custom_section_5"] = [
				"permission" => $permissions["content_custom_section_5"],
				"permission_key" => "content_custom_section_5",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_5"],
				"label_original" => "Custom Section 5 (Web Only)",
				"label_key" => "custom_section_5",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_5_show_date_added"]
			];

			$settings["content"]["custom_section_6"] = [
				"permission" => $permissions["content_custom_section_6"],
				"permission_key" => "content_custom_section_6",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["custom_section_6"],
				"label_original" => "Custom Section 6 (Web Only)",
				"label_key" => "custom_section_6",
				"label_editable" => true,
				"show_date_added" => $sms_settings["custom_section_6_show_date_added"]
			];
			
			// Informational features.
			
			$settings["information"]["about"] = [
				"permission" => $permissions["us_about"],
				"permission_key" => "us_about",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["about"],
				"label_original" => "About Us",
				"label_key" => "about",
				"label_editable" => true,
			];
			
			$settings["information"]["services"] = [
				"permission" => $permissions["us_services"],
				"permission_key" => "us_services",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["services"],
				"label_original" => "Services",
				"label_key" => "services",
				"label_editable" => true,
			];
			
			$settings["information"]["team"] = [
				"permission" => $permissions["us_team"],
				"permission_key" => "us_team",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["team"],
				"label_original" => "Team Member",
				"label_key" => "team",
				"label_editable" => true,
			];
			
			$settings["information"]["locations"] = [
				"permission" => $permissions["us_locations"],
				"permission_key" => "us_locations",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["locations1"],
				"label_original" => "Locations",
				"label_key" => "locations",
				"label_editable" => true,
			];
			
			$settings["information"]["enquiries"] = [
				"permission" => $permissions["us_enquiries"],
				"permission_key" => "us_enquiries",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["enquiries"],
				"label_original" => "Enquiries",
				"label_key" => "enquiries",
				"label_editable" => true,
			];
			
			// Fan wall.
			
			$settings_fan_wall = Client::local("settings_fan_wall")->select("recipient_address")->first();
			
			if (strlen($settings_fan_wall->recipient_address)) {
				$fan_wall_recipients = explode(",", $settings_fan_wall->recipient_address);
			} else {
				$fan_wall_recipients = [];
			}
			
			$settings["fan_wall"] = [
				
				"permission" => $permissions["content_fan_wall"],
				"permission_key" => "content_fan_wall",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["fan_wall"],
				"label_original" => "Fan Wall",
				"label_key" => "fan_wall",
				"label_editable" => true,
				
				"recipients" => $fan_wall_recipients,
				
				"compose_fan_post" => $labels["compose_fan_post"],
				
			];
			
			// Events.
			
			$settings_events = Client::local("settings_events")->select("recipient_address")->first();
			
			if (strlen($settings_events->recipient_address)) {
				$rsvp_recipients = explode(",", $settings_events->recipient_address);
			} else {
				$rsvp_recipients = [];
			}
			
			$settings["events"] = [
				
				"permission" => $permissions["content_events"],
				"permission_key" => "content_events",
				"permission_editable" => $diay_only ? false : true,
				
				"label" => $labels["events"],
				"label_original" => "Events",
				"label_key" => "events",
				"label_editable" => true,
				
				"rsvp_recipients" => $rsvp_recipients,
				"rsvp_permission" => $permissions["content_rsvp"],
				"rsvp_permission_key" => "content_rsvp",
				"rsvp_label" => "RSVP",
				
			];
			
			// Giving.
			
			$settings["giving"] = [
				
				"permission" => $permissions["us_giving"],
				"permission_key" => "us_giving",
				"permission_editable" => $diay_only ? false : true,
				"label" => $labels["giving"],
				"label_original" => "Giving",
				"label_key" => "giving",
				"label_editable" => true,
				
				"url" => Client::local("settings_giving")->select("url")->value("url"),
				
			];
			
			// Shop.
			
			$settings["shop"] = [
				
				"permission" => $permissions["us_shop"],
				"permission_key" => "us_shop",
				"permission_editable" => $diay_only ? false : true,
				
				"label" => $labels["shop"],
				"label_original" => "Shop",
				"label_key" => "shop",
				"label_editable" => true,
				
				"url" => Client::local("settings_shop")->select("url")->value("url"),
				
			];
			
		}
		
		// Premium features.
		
		$settings["premium"]["user_groups"] = [
			"permission" => $permissions["us_user_groups"],
			"label" => $labels["user_groups"],
			"label_key" => "user_groups",
		];
		
		$settings["premium"]["manual_push"] = [
			"permission" => $permissions["manual_push"],
			"label" => "Manual Push",
		];
		
		$settings["premium"]["automatic_push"] = [
			"permission" => $permissions["automatic_push"],
			"label" => "Automatic Push",
		];
		
		$settings["premium"]["v_social"] = [
			"permission" => $permissions["publish_to_social_media"],
			"label" => "V Social",
		];
		
		$settings["premium"]["analytics"] = [
			"permission" => $permissions["dashboard_analytics"],
			"label" => "Analytics",
		];

		// Content Filters.
		
		$settings["filters"]["location_state_filter"] = [
			"permission" => $permissions["location_state_filter"],
			"label" => "Locations States Filter",
			"permission_key" => "location_state_filter",
			"permission_editable" => $diay_only ? false : true
		];

		$settings["filters"]["sermon_filter"] = [
			"permission" => $permissions["sermon_filter"],
			"label" => "Sermons Filter",
			"permission_key" => "sermon_filter",
			"permission_editable" => $diay_only ? false : true
		];
		
		// Response.
		
		return $response->withJson((new \Vinteract\API\Response([
			"code" => 200,
			"contents" => [ "settings" => $settings ],
		]))->getResponseObject());
		
	});
	
	/*
		Run App
	*/
	
	$app->run();
	