<?php

namespace Vinteract\Payments;

use \Vinteract\Client;
use \Vinteract\Utils;
use \Vinteract\Features;

use DirectoryIterator;

class Payments {
	
	const NAME = "Redde";
	
	const SLOGAN = "Payments Made Easy";
	
	const WEBSITE_URL = "https://www.redde.com.au/";
	
	const WEBSITE_DOMAIN = "redde.com.au";
	
	const DEFAULT_PROVIDER = "PinPayments";
	
	const DEFAULT_CURRENCY = "AUD";

	CONST PLATFORM_FEES = [
		"church" => [
			"domestic" => [ "percentage" => 1.9, "fixed" => 30 ],
			"international" => [ "percentage" => 2.9, "fixed" => 30 ]
		],
		"business" => [
			"domestic" => [ "percentage" => 1.9, "fixed" => 30 ],
			"international" => [ "percentage" => 2.9, "fixed" => 30 ]
		],
	];
	
	private static $PROVIDER_NAME;
	
	private static $PROVIDER_CLASS;
	
	public static function boot() {
		
		// Create default global settings.
		
		if (!Client::global("settings")->where("key", "vpay_application_form_verto_group_password")->exists()) {
			Client::global("settings")->insert([ "key" => "vpay_application_form_verto_group_password", "value" => \Vinteract\Utils::generatePublicId() ]);
		}
		
		if (!Client::global("settings")->where("key", "vpay_application_form_pin_payments_password")->exists()) {
			Client::global("settings")->insert([ "key" => "vpay_application_form_pin_payments_password", "value" => \Vinteract\Utils::generatePublicId() ]);
		}
		
		// Create default account information.
		
		if (!Client::local("vpay_account")->where("key", "status")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "status", "value" => "inactive" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "status_message")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "status_message", "value" => "" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "platform_fee_percentage")->exists()) {
			if (Client::isChurch()) {
				Client::local("vpay_account")->insert([ "key" => "platform_fee_percentage", "value" => self::PLATFORM_FEES["church"]["domestic"]["percentage"] ]);
			} else {
				Client::local("vpay_account")->insert([ "key" => "platform_fee_percentage", "value" => self::PLATFORM_FEES["business"]["domestic"]["percentage"] ]);
			}
		}
		
		if (!Client::local("vpay_account")->where("key", "platform_fee_fixed")->exists()) {
			if (Client::isChurch()) {
				Client::local("vpay_account")->insert([ "key" => "platform_fee_fixed", "value" => self::PLATFORM_FEES["church"]["domestic"]["fixed"] ]);
			} else {
				Client::local("vpay_account")->insert([ "key" => "platform_fee_fixed", "value" => self::PLATFORM_FEES["business"]["domestic"]["fixed"] ]);
			}
		}
		
		if (!Client::local("vpay_account")->where("key", "number_of_transactions_per_month")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "number_of_transactions_per_month", "value" => "400" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "average_sale_amount")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "average_sale_amount", "value" => "100" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "total_annual_turnover")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "total_annual_turnover", "value" => "0" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "average_days_to_deliver")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "average_days_to_deliver", "value" => "0" ]);
		}
		
		// Create default settings.
		
		if (!Client::local("vpay_settings")->where("key", "provider")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "provider", "value" => self::DEFAULT_PROVIDER ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "default_currency")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "default_currency", "value" => self::DEFAULT_CURRENCY ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "disable_payment_form")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "disable_payment_form", "value" => "1" ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "disable_payment_form_message")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "disable_payment_form_message", "value" => "We are currently getting things setup... please bear with us!" ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "logo_url")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "logo_url", "value" => "" ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "brand_colour")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "brand_colour", "value" => "" ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "hosted_page_brand_name")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "hosted_page_brand_name", "value" => Client::getDisplayName() ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "sandbox_mode")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "sandbox_mode", "value" => "true" ]);
		}
		
		if (!Client::local("vpay_settings")->where("key", "receipt_bcc_recipients")->exists()) {
			Client::local("vpay_settings")->insert([ "key" => "receipt_bcc_recipients", "value" => "" ]);
		}
		
		// Load the provider.
		
		self::loadProvider();
		
		// Register aliases for the provider's classes.
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Transaction") and !class_exists("Vinteract\Payments\Transaction")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Transaction", "Vinteract\Payments\Transaction");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\RecurringTransaction") and !class_exists("Vinteract\Payments\RecurringTransaction")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\RecurringTransaction", "Vinteract\Payments\RecurringTransaction");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Refund") and !class_exists("Vinteract\Payments\Refund")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Refund", "Vinteract\Payments\Refund");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\MerchantAccount") and !class_exists("Vinteract\Payments\MerchantAccount")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\MerchantAccount", "Vinteract\Payments\MerchantAccount");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Balance") and !class_exists("Vinteract\Payments\Balance")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Balance", "Vinteract\Payments\Balance");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\CardToken") and !class_exists("Vinteract\Payments\CardToken")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\CardToken", "Vinteract\Payments\CardToken");
		}
		
		if (class_exists("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Transfers\TransferRequest") and !class_exists("Vinteract\Payments\TransferRequest")) {
			class_alias("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\Transfers\TransferRequest", "Vinteract\Payments\TransferRequest");
		}
		
	}
	
	public static function loadProvider() {
		
		// Store the provider's name.
		
		self::$PROVIDER_NAME = self::getProviderName();
		
		// Store the provider's class name.
		
		self::$PROVIDER_CLASS = ("\Vinteract\Payments\Providers\\" . self::$PROVIDER_NAME . "\\" . self::$PROVIDER_NAME);
		
		// Boot the provider.
		
		self::$PROVIDER_CLASS::boot();
		
	}
	
	public static function getTermsAndConditionsTemplateText($variables = []) {
		
		$template = [
			
			"<p>" . self::getName() . " is a V interact Product owned by Verto Group Pty Ltd (c) August 2019.</p>",
			
			"<h3>Introduction</h3>",
			
			"<p>Thank you for choosing to give to {CLIENT_NAME}. By making this transaction you are agreeing to be bound by the following Payment Terms and Conditions.</p>",
			"<p>{CLIENT_NAME} regards customer privacy as an important part of our relationship with our customers. The following Payment Terms and Conditions apply to all users across our web and mobile applications and conforms to Internet privacy standards. Please also refer to our Privacy Policy <a target='blank' href='{CLIENT_PRIVACY_POLICY_URL}'>here</a> and our User Agreement <a target='blank' href='{CLIENT_USER_AGREEMENT_URL}'>here</a> which will be updated from time to time.</p>",
			
			"<p>All payments are facilitated under a software license to our technology provider, Verto Group Pty Ltd (ACN 160 166 727) who own the V interact Platform including " . self::getName() . " (referred to as “V interact” or the “Software”). By the term “Software“ we mean the platforms we license including but not limited to our website and mobile applications. You can find Verto Group’s privacy policy <a target='blank' href='{VERTO_PRIVACY_POLICY_URL}'>here</a>. The payments are processed by a third-party payment processor (TPPP) under <a target='blank' href='{PAYMENT_PROCESSOR_TERMS_URL}'>Terms of Service</a> between {CLIENT_NAME} and that TPPP.</p>",
			
			"<p>These Payment Terms and Conditions may be updated and changed without notice.</p>",
			
			"<h3>Collection of Information</h3>",
			
			"<p>In order to make payments, we may require information from you in order to provide the best service possible. All correspondence may also be collected and stored, particularly in regard to sales, support and accounts, including email.</p>",
			
			"<p>Any information collected by {CLIENT_NAME} is collected via correspondence from you or your company. This may be via the telephone, email, mail, fax or directly through the Software. Your personal information, such as your name, address and email address is collected in accordance with our privacy policy.</p>",
			
			"<p>All financial information collected is used by our TPPP in order to process once-off or recurring payments from you to {CLIENT_NAME}. You can access our TPPP’s privacy policy <a target='blank' href='{PAYMENT_PROCESSOR_PRIVACY_POLICY_URL}'>here</a>.</p>",
			
			"<h3>Use of and access to Collected Information</h3>",
			
			"<p>Any information that we collect from you is used in accordance with our Privacy Policy and User Agreement. Any details we collect from our users are required in order to provide you with our products and/or services, and a high level of customer service. These Payment Terms govern how your financial information is used for the processing of payments through V interact.</p>",
			
			"<p>If your personally identifiable information changes, or if you no longer desire our service, you may correct, update, delete or deactivate it by contacting us by email or phone or if relevant, access the subscriptions section within My Profile on V interact.</p>",
			
			"<h3>Storage of Collected Information</h3>",
			
			"<p>The security of your personal information is important to us. When you enter sensitive information (such as personal information) on our website, we encrypt that information using secure socket layer technology (SSL). For information on how Credit Card details are collected and stored, please see our Security Policy below.</p>",
			
			"<p>We follow generally accepted industry standards to protect the personal information submitted to us, both during transmission and once we receive it.</p>",
			
			"<p>If you have any questions about the security of the Software (Website and Mobile Applications), please contact us.</p>",
			
			"<h3>Processing of payments</h3>",
			
			"<p>{CLIENT_NAME} can accept one-off or recurring payments for donations, subscriptions, events, products or any other goods or service that we may offer from time to time. Payments can be processed securely through V interact as detailed below. Transaction fees may be applicable to each payment you make through the Software. All payments are subject to our Security Policy.</p>",
			
			"<p style='margin-left: 30px; font-weight: bold;'>a) One-off Payments</p>",
			
			"<p>For one-off payments, your personal information such as name, email address, mobile number, and credit card information is collected. Address details may also be collected. We do not have access to your financial information – our TPPP is the only party that can see your full credit card number and uses it solely to process payments on our behalf.</p>",
			
			"<p style='margin-left: 30px; font-weight: bold;'>b) Recurring Payments</p>",
			
			"<p>For recurring payments, our TPPP holds your personal information (name, email address and credit card) for the purpose of processing recurring payments. Our technology provider (Verto Group) schedules the recurring payments through the use of encryption. The credit card number never touches, or is stored, on Verto Groups server.</p>",
			
			"<p>If you elect to initiate recurring payments (which includes recurring donations), you are authorizing us to store and automatically charge your selected method of payment in the payment amount (including fees and charges) and frequency that you designated, without further consent or instructions from you. If you wish to cancel your recurring payments, you may do so by prior to the next scheduled payment date under My Profile.</p>",
			
			"<p>If for any reason, the Software becomes temporarily unavailable, any recurring payments scheduled at that time will not be processed, and you will receive a notification of your next scheduled recurring payment from us (or Verto Group on our behalf).</p>",
			
			"<p>If any changes are made to a stored payment method or linked account is changed, any recurring payments associated with that stored payment method or linked account may be paused, cancelled or otherwise adversely affected. Cancelling an authorization for a specific recurring payment schedule will not cancel other authorizations for recurring payments.</p>",
			
			"<p>If you schedule a recurring payment, the recurring payment amount you specified will be billed to the credit or debit card, bank account, or other account you designate during the setup process. If you want to designate a different credit, debit card or bank or other account, or if there is a change in such cards or accounts, you must cancel your existing schedule payment through My Profile and then setup a new recurring payment. This may temporarily delay your ability to make online payments while your new payment information is verified.</p>",
			
			"<h3>Security Policy</h3>",
			
			"<p>All payments are processed securely through " . self::getName() . ", which uses a fully accredited, secure and trusted TPPP to process all online payments. All online credit card transactions performed on this site using the Pin Payments gateway are secured payments.</p>",
			
			"<p>* Payments are processed in real time with an immediate response.</p>",
			"<p>* All transactions are submitted across a secure, encrypted connection.</p>",
			"<p>* Your complete credit card number cannot be viewed by {CLIENT_NAME} or any outside party.</p>",
			"<p>* The TPPP is PCI-DSS Level 1 compliant, and a registered payment service provider with Visa and Mastercard.</p>",
			
			"<p>For more information about Pin Payments and online credit card payments, please <a target='blank' href='{PAYMENT_PROCESSOR_TERMS_URL}'>click here</a>.</p>",
			
			"<h3>Managing Your User Account</h3>",
			
			"<ul>",
			"<li>update your personal details at any time within the “My Profile” settings;</li>",
			"<li>delete your user account, including any preferences and subscriptions you are running by selecting ‘delete’ within the My Profile section.</li>",
			"<li>edit your personal content preferences (for free content) at any time within the My Profile section; and</li>",
			"<li>cancel your recurring payment by selecting ‘cancel’ via Subscriptions or Recurring within the My Profile section.</li>",
			"</ul>",
			
			"<h3>Third Parties</h3>",
			
			"<p>{CLIENT_NAME} may at its discretion use other third parties to provide essential services on our site or for our business processes. We may share your details as necessary for the third party to provide that service. These third parties are prohibited from using your personally identifiable information for any other purpose unless where you have provided your consent.</p>",
			
			"<p>{CLIENT_NAME} does not share any information with third parties for any unknown or unrelated uses.</p>",
			
			"<h3>Legal</h3>",
			
			"<p>We reserve the right to disclose your personally identifiable information as required by law and when we believe that disclosure is necessary to protect our rights and/or comply with a judicial proceeding, court order, or legal process served on the Software.</p>",
			
			"<h3>Delivery Policy</h3>",
			
			"<p>After ordering online, you will receive an email confirmation containing your order details (if you have provided your email address), normally within a few minutes of ordering.</p>",
			
			"<p>For access to content or content subscriptions, you will gain access to this content as soon as your payment transaction is approved and an account is created and activated. Subscriptions we offer may require either a recurring or once off payment. You acknowledge and agree that where access to content is required under a recurring subscription, your access to that content is contingent upon the payment being received. If the recurring payment is not received, your access to the content and subscription will be cancelled;</p>",
			
			"<p>We may change, suspend or end any content, subscription, or change and modify prices prospectively in our discretion. To the extent allowed under law, these changes may be effective upon notice provided to you.</p>",
			
			"<p>You agree and understand that you can cancel your paid subscription by selecting ‘cancel subscription’ via Subscriptions within the My Profile section. All associated content for this subscription will be inaccessible from the Service from the date the paid subscription expires. Some content will be permanently deleted. This information cannot be recovered once it has been permanently deleted.</p>",
			
			"<p>For products ordered, we will attempt to send your goods via parcel post within 14 working days; however if goods are unavailable, delivery will take a little longer.</p>",
			
			"<p>If you wish to query a delivery, please contact us by calling [insert number] or emailing us at [insert email address].</p>",
			
			"<h2>Disclaimer and Limit of Liability</h2>",
			
			"<h3>No Warranty</h3>",
			
			"<p>This is our disclaimer of legal liability for the quality, safety, or reliability of our Services.</p>",
			
			"<p>To the extent allowed under law, we, (and those that we work with, including our technology partner Verto Group whom we license the Software from) (a) disclaim all implied warranties and representations (e.g. warranties of merchantability, fitness for a particular purpose, accuracy of data, and noninfringement); (b) do not guarantee that the services will function without interruption or errors, and (c) provide the Software (including content and information) on an “as is” and “as available” basis.</p>",
			
			"<h3>Exclusion of Liability</h3>",
			
			"<p>To the extent permitted under law, we, (and those that we work with, including our technology partner Verto Group whom we license the software from) shall not be liable to you or others for any indirect, incidental, special, consequential or punitive damages, or any loss of data, opportunities, reputation, profits or revenues, related to the Software (e.g. offensive, illegal or defamatory statements, down time or loss, use or changes to your information or content).</p>",
			
			"<p>Disclaimer of Verto Group’s Liability. Verto Group is not responsible for any aspect of the products or services you purchase or for any communications or offers made to you. Verto Group’s role as the software licensor for enabling payments to be received by {CLIENT_NAME} will not make Verto Group liable or responsible for breach of representations and warranties, non-provision of after-sales or warranty services or fraud vis-a-vis the products and/or services sold by a merchant.</p>",
			
			"<h3>Refund & Returns Policy</h3>",
			
			"<p>Donations: There are no refunds for donations.</p>",
			
			"<p>Subscriptions: There are no refunds for Subscriptions on the Software.</p>",
			
			"<p>Products: If for any reason you are not completely satisfied with your purchase we will give you a 10 day money-back guarantee from the time you receive the goods. Please email us within that time if you are not satisfied with your purchase so that we can resolve any problems.</p>",
			
			"<p>This refund policy does not apply to goods which have been worn or used, damaged after delivery, or if any attempt has been made to alter the product or if they have been dropped or broken. All products must be returned in their original condition. All postage and insurance costs are to be paid by the buyer. We recommend that you return the product via Registered post and that you pre-pay all postage. You assume any risk of loss, theft or damaged goods during transit & we therefore advise you take out shipment registration of insurance with your postal carrier. {CLIENT_NAME} will not be responsible for parcels lost or damaged in transit if you choose not to insure.</p>",
			
			"<h3>Complaints handling & Feedback</h3>",
			
			"<p>We endeavour to resolve all complaints in a timely manner. If you have a complaint or enquiry, wish to give feedback or update / amend your personal information, please contact:</p>",
			
			"<p>[insert name of contact person or title e.g.]</p>",
			
			"<p>Privacy Officer</p>",
			"<p>[insert name]</p>",
			"<p>[insert phone number]</p>",
			"<p>[insert email address]</p>",
			
		];
		
		$variables = [
			"{CLIENT_NAME}" => self::getAccountDetail("legal_entity_name"),
			"{CLIENT_PRIVACY_POLICY_URL}" => \Vinteract\Client::getWebsiteURL("privacy_policy"),
			"{CLIENT_USER_AGREEMENT_URL}" => \Vinteract\Client::getWebsiteURL("user_agreement"),
			"{VERTO_PRIVACY_POLICY_URL}" => "https://vertogroup.com.au/privacy-policy/",
			"{PAYMENT_PROCESSOR_TERMS_URL}" => self::$PROVIDER_CLASS::getTermsAndConditionsUrl(),
			"{PAYMENT_PROCESSOR_PRIVACY_POLICY_URL}" => self::$PROVIDER_CLASS::getPrivacyPolicyUrl(),
		];
		
		return \Vinteract\Utils::replaceVariablesInString(join("", $template), $variables);
		
	}
	
	public static function getAccountStatus() {
		return self::$PROVIDER_CLASS::getAccountStatus();
	}
	
	public static function getAccountStatusMessage() {
		return self::$PROVIDER_CLASS::getAccountStatusMessage();
	}
	
	public static function getAccountDetail($key = "") {
		return Client::local("vpay_account")->where("key", $key)->value("value");
	}
	
	public static function getInformation($keys = []) {
		return array_column(Client::local("vpay_account")->whereIn("key", $keys)->get()->toArray(), "value", "key");
	}
	
	public static function getSetting($key = "") {
		return Client::local("vpay_settings")->where("key", $key)->value("value");
	}
	
	public static function getSettings($keys = []) {
		return array_column(Client::local("vpay_settings")->whereIn("key", $keys)->get()->toArray(), "value", "key");
	}
	
	public static function getIdentityDocuments() {
		
		$client_id = Client::getPublicId();
		
		$uploads_path = "uploads/clients/{$client_id}/identification";
		
		$files = [];
		
		if (is_dir($uploads_path)) {
			
			$directory = new DirectoryIterator(\Vinteract\Utils::getPaymentsApiPath($uploads_path));
			
			foreach ($directory as $fileInfo) {
				
				if ($fileInfo->isDot()
					or $fileInfo->isDir()
					or $fileInfo->getBasename() === ".DS_Store"
					or $fileInfo->getFilename() === "index.html") {
					continue;
				};
				
				$filename = $fileInfo->getFilename();
				
				$url = \Vinteract\Utils::getPaymentsApiUrl("{$uploads_path}/{$filename}");
				
				$files[] = [
					"name" => $filename,
					"path" => $fileInfo->getPath() . "/" . $filename,
					"size" => $fileInfo->getSize(),
					"url" => $url,
				];
				
			}
			
		}
		
		return $files;
		
	}
	
	public static function getAccountDetails($params = []) {
		
		return [
			
			"information" => self::getInformation(array_key_exists("information", $params) ? $params["information"] : []),
			
			"settings" => self::getSettings(array_key_exists("settings", $params) ? $params["settings"] : []),
			
			"trustees" => \Vinteract\Payments\AccountTrustee::all(),
			
			"personnel" => \Vinteract\Payments\AccountPersonnel::all(),
			
			"documents" => self::getIdentityDocuments(),
			
			"sandbox" => self::isSandboxMode(),
			
			"status" => self::getAccountStatus(),
			"status_message" => self::getAccountStatusMessage(),
			
		];
		
	}
	
	public static function getProviderDetails() {
		
		return [
			"name" => self::$PROVIDER_CLASS::getName(),
			"balance" => self::$PROVIDER_CLASS::getAccountBalance(),
			"urls" => self::$PROVIDER_CLASS::getUrls(),
			"settings" => self::$PROVIDER_CLASS::getSettings(),
		];
		
	}
	
	public static function getPlatformDetails() {
		
		// Fees and settlement.
		
		$churchFeesDomestic = self::PLATFORM_FEES["church"]["domestic"];
		$churchFeesInternational = self::PLATFORM_FEES["church"]["international"];
		
		$businessFeesDomestic = self::PLATFORM_FEES["business"]["domestic"];
		$businessFeesInternational = self::PLATFORM_FEES["business"]["international"];
		
		if (Client::isChurch()) {
			$domesticRate = ($churchFeesDomestic["percentage"] . "% + " . Utils::formatMoney($churchFeesDomestic["fixed"] / 100, true));
			$internationalRate = ($churchFeesInternational["percentage"] . "% + " . Utils::formatMoney($churchFeesInternational["fixed"] / 100, true));
		} else {
			$domesticRate = ($businessFeesDomestic["percentage"] . "% + " . Utils::formatMoney($businessFeesDomestic["fixed"] / 100, true));
			$internationalRate = ($businessFeesInternational["percentage"] . "% + " . Utils::formatMoney($businessFeesInternational["fixed"] / 100, true));
		}
		
		$feesAndSettlement = [
			[
				"card_type" => "Domestic",
				"description_short" => "Domestic VISA MCARD",
				"description_long" => "Domestic VISA MasterCard",
				"rate" => $domesticRate,
				"chargebacks" => "$25.00",
				"settlement" => "1 Day"
			],
			[
				"card_type" => "International",
				"description_short" => "International",
				"description_long" => "International Cards",
				"rate" => $internationalRate,
				"chargebacks" => "$25.00",
				"settlement" => "1 Day"
			],
			[
				"card_type" => "American Express",
				"description_short" => "American Express",
				"description_long" => "American Express",
				"rate" => "2.25% + $0.30",
				"chargebacks" => "$25.00",
				"settlement" => "1 Day"
			],
		];
		
		// Return platform information.
		
		return [
			"name" => self::getName(),
			"slogan" => self::getSlogan(),
			"website_url" => self::getWebsiteURL(),
			"fees_and_settlement" => $feesAndSettlement,
			"logo" => [
				"large" => self::getAssetURL("img/logo-large.png"),
				"small" => self::getAssetURL("img/logo-small.png"),
				"slogan-large" => self::getAssetURL("img/logo-slogan-large.png"),
				"slogan-small" => self::getAssetURL("img/logo-slogan-small.png"),
				"payments-large" => self::getAssetURL("img/logo-payments-large.png"),
				"payments-small" => self::getAssetURL("img/logo-payments-small.png"),
				"payments-by-large" => self::getAssetURL("img/logo-payments-by-large.png"),
				"payments-by-small" => self::getAssetURL("img/logo-payments-by-small.png"),
			],
		];
		
	}
	
	public static function getName() {
		return self::NAME;
	}
	
	public static function getSlogan() {
		return self::SLOGAN;
	}
	
	public static function getWebsiteURL($extra = "") {
		return self::WEBSITE_URL . $extra;
	}
	
	public static function getAssetURL($asset_path = "") {
		return self::WEBSITE_URL . "assets/" . $asset_path;
	}
	
	public static function getWidgetURL($widget_path = "") {
		return self::WEBSITE_URL . "widgets/" . $widget_path;
	}
	
	public static function getCurrency() {
		return self::DEFAULT_CURRENCY;
	}
	
	public static function getAccessToken($key = "") {
		return \Vinteract\Client::global("payments_access_tokens")
			->where("org_type", \Vinteract\Client::getType())
			->where("org_id", \Vinteract\Client::getId())
			->where("key", $key)
			->value("access_token");
	}
	
	public static function getApplicationFormURL() {
		return self::getWebsiteURL("application/?id=" . self::getAccessToken("client_portal"));
	}
	
	public static function getReceiptURL($receiptNumber = "") {
		if (Client::ready()) {
			$urlPath = "payments/receipt/{$receiptNumber}";
			if (Features::isFeatureEnabled("vinteract_pay_hosted_page")) {
				$clientDomain = Client::getData("payments_subdomain");
				$hostDomain = self::WEBSITE_DOMAIN;
				return "https://{$clientDomain}.{$hostDomain}/{$urlPath}";
			}
			return Client::getWebsiteURL($urlPath);
		}
		return "";
	}
	
	public static function getLogoURL() {
		return self::getSetting("logo_url");
	}
	
	public static function getProviderName() {
		return self::getSetting("provider");
	}
	
	public static function getFees() {
		return [
			"percentage" => self::getSetting("platform_fee_percentage"),
			"fixed" => self::getSetting("platform_fee_fixed"),
		];
	}
	
	public static function getUrl($key) {
		return self::$PROVIDER_CLASS::getUrl($key);
	}
	
	public static function getReceiptBccRecipients() {
		
		$recipients = self::getSetting("receipt_bcc_recipients");
		
		if (isset($recipients)) {
			$exploded = explode(",", $recipients);
			return array_map(function($email) {
				return [
					"email" => $email,
					"name" => $email,
					"type" => "bcc",
				];
			}, $exploded);
		}
		
		return [];
		
	}
	
	public static function generateReddeOfficeEmailAddress($clientName = "") {
		
		$caseTransformed = strtolower($clientName);
		
		$exploded = explode(" ", $caseTransformed);
		
		$firstTwoWords = array_slice($exploded, 0, 2);
		
		$firstTwoWordsJoined = implode(".", $firstTwoWords);
		
		return "office+{$firstTwoWordsJoined}@" . self::WEBSITE_DOMAIN;
		
	}
	
	public static function calculateFullFeeOnAmount($amount = 0) {
		
		$platform_fees = self::getFees();
		
		if (count($amount)) {
			$amount = ($amount * ($platform_fees["percentage"] / 100) + ($platform_fees["fixed"] / 100));
		}
		
		return \Vinteract\Utils::formatMoney($amount);
		
	}
	
	public static function calculatePlatformFeeOnAmount($amount = 0) {
		
		$platform_fees = self::getFees();
		$provider_fees = self::$PROVIDER_CLASS::getFees();
		
		$difference_percentage = $platform_fees["percentage"] - $provider_fees["percentage"];
		$difference_fixed = $platform_fees["fixed"] - $provider_fees["fixed"];
		
		if (count($amount)) {
			$amount = ($amount * ($difference_percentage / 100) + ($difference_fixed / 100));
		}
		
		return \Vinteract\Utils::formatMoney($amount);
		
	}
	
	public static function isSandboxMode() {
		return Client::local("vpay_settings")->where("key", "sandbox_mode")->value("value") == "true";
	}
	
}
