<?php

namespace Vinteract\Payments\Providers\PinPayments;

use \Vinteract\Client as Client;

use \Vinteract\Payments\Providers\PaymentProvider;

use DateTime;
use DateTimeZone;

class PinPayments extends PaymentProvider {
	
	const NAME = "PinPayments";
	
	const SANDBOX_API_URL = "https://test-api.pinpayments.com";
	
	const API_URL = "https://api.pinpayments.com";
	
	const TERMS_AND_CONDITIONS_URL = "https://www.reddepay.com.au/assets/terms-and-conditions/pin-payments/index.html";
	
	const MERCHANT_AGREEMENT_URL = "https://www.reddepay.com.au/assets/merchant-agreement/pin-payments/index.html";
	
	public static function boot() {
		
		// Create settings.
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_secret_key")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_secret_key", "value" => "" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_secret_key_sandbox")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_secret_key_sandbox", "value" => "" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_publishable_key")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_publishable_key", "value" => "" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_publishable_key_sandbox")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_publishable_key_sandbox", "value" => "" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_fee_percentage")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_fee_percentage", "value" => "1.0" ]);
		}
		
		if (!Client::local("vpay_account")->where("key", "pin_payments_fee_fixed")->exists()) {
			Client::local("vpay_account")->insert([ "key" => "pin_payments_fee_fixed", "value" => "10" ]);
		}
		
	}
	
	public static function getAccountStatus() {
		return Client::local("vpay_account")->where("key", "status")->value("value");
	}
	
	public static function getAccountStatusMessage() {
		return Client::local("vpay_account")->where("key", "status_message")->value("value");
	}
	
	public static function getAccountBalance() {
		
		// Variables.
		
		$tomorrowsDate = (new DateTime("tomorrow", new DateTimeZone(Client::getTimezone())))->format("Y-m-d");
		$todaysDate = (new DateTime("now", new DateTimeZone(Client::getTimezone())))->format("Y-m-d");
		
		// Get balance.
		
		$balanceResponse = (new \Vinteract\Payments\Balance())->process();
		
		$originalTotal = $balanceResponse->getValue("data.pending");
		$total = ($originalTotal[0]["amount"] / 100);
		
		// Get transferred funds.
		
		$transfersResponse = (new \Vinteract\Payments\TransferRequest())->process(["from" => $tomorrowsDate, "to" => $todaysDate]);
		
		$transferred = 0;
		
		foreach ($transfersResponse->getItems() as $transfer) {
			$total += $transfer->getAmount();
			$transferred += $transfer->getAmount();
		}
		
		// Return response.
		
		return [
			"enabled" => true,
			"total" => $total,
			"cleared" => $transferred,
			"uncleared" => ($total - $transferred),
		];
		
	}
	
	public static function getName() {
		return self::NAME;
	}
	
	public static function getUrls() {
		return [
			"api" => self::getApiUrl(),
			"terms_and_conditions" => self::TERMS_AND_CONDITIONS_URL,
			"merchant_agreement" => self::MERCHANT_AGREEMENT_URL,
		];
	}
	
	public static function getUrl($key) {
		return self::getUrls()[$key];
	}
	
	public static function getSettings() {
		return [
			"publishable_key" => self::getPublishableKey(),
		];
	}
	
	public static function getSecretKey() {
		$key = "pin_payments_secret_key";
		if (\Vinteract\Payments\Payments::isSandboxMode()) {
			$key = "pin_payments_secret_key_sandbox";
		}
		return Client::local("vpay_account")->where("key", $key)->value("value");
	}
	
	public static function getPublishableKey() {
		$key = "pin_payments_publishable_key";
		if (\Vinteract\Payments\Payments::isSandboxMode()) {
			$key = "pin_payments_publishable_key_sandbox";
		}
		return Client::local("vpay_account")->where("key", $key)->value("value");
	}
	
	public static function getRequestUrl($endpoint = "") {
		return self::getApiUrl("1/{$endpoint}");
	}
	
	public static function getApiUrl($extra = "") {
		
		$url = self::API_URL;
		
		if (\Vinteract\Payments\Payments::isSandboxMode()) {
			$url = self::SANDBOX_API_URL;
		}
		
		if ($extra != "") {
			return "{$url}/{$extra}";
		}
		
		return $url;
		
	}
	
	public static function getFees() {
		return [
			"percentage" => Client::local("vpay_account")->where("key", "pin_payments_fee_percentage")->value("value"),
			"fixed" => Client::local("vpay_account")->where("key", "pin_payments_fee_fixed")->value("value"),
		];
	}
	
	public static function getTermsAndConditionsUrl() {
		return "https://www.pinpayments.com/terms";
	}
	
	public static function getPrivacyPolicyUrl() {
		return "https://www.pinpayments.com/privacy";
	}
	
}
